# ===========================================================================
# File: convert.tcl
# Target: trac
#                        Created: 2010-08-29 09:51:41
#              Last modification: 2013-12-23 17:35:14
# Author: Bernard Desgraupes
# e-mail: <bdesgraupes@users.sourceforge.net>
# (c) Copyright: Bernard Desgraupes 2010-2013
# All rights reserved.
# Description: Aida callbacks for target trac
# ===========================================================================


namespace eval trac {

	# Ensure fallback on base commands
	namespace path ::base
	
}


# Hooks
# -----

proc trac::preConvertHook {} {}
proc trac::postConvertHook {} {}

proc trac::splitHook {file} {}


# Callbacks
# ---------


## 
 # ------------------------------------------------------------------------
 # 
 # "trac::anchorProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::anchorProc {label} {
	set label [string trim $label "\"' "]
	return "\[=#$label\]"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "trac::commentProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::commentProc {str} {
	return "{{{\n#!comment $str\n}}}"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "trac::horizRuleProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::horizRuleProc {} {
	return "----"
}


##
 # ------------------------------------------------------------------------
 # 
 # "trac::imageProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::imageProc {str attr} {
	set attrDict [aida::getAttr img $attr]
	set str [string trim $str "\"' "]
	return "\[\[Image($str, [aida::dictToAttrString $attrDict])\]\]"
}


##
 # ------------------------------------------------------------------------
 # 
 # "trac::linkProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::linkProc {str url} {
	set str [string trim $str]
	if {$str eq ""} {
		return $url
	} else {
		return "\[\[$url| $str\]\]"
	} 
}


## 
 # ------------------------------------------------------------------------
 # 
 # "trac::listProc" --
 # 
 # The 'type' attribute can be "1", "a", "A", "i", "I".
 # The 'start' attribute is supported only if the 'type' attribute is "1".
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::listProc {kind depth attr itemList} {
	set attrDict [aida::getAttr $kind $attr]
	set block [list]

	if {[catch {dict get $attrDict start} cnt]} {
		set cnt 1
	} 
	if {[catch {dict get $attrDict type} tp]} {
		set tp "1"
	} 
	set indent [string repeat " " $depth]
	set marker "<@t@r@c@>"
	set prefix "$marker$indent"

	switch -- $kind {
		"ol" {
			if {$tp eq "1"} {
				set head "$cnt. "
			} else {
				set head "$tp. "
			} 
		}
		"ul" {
			set head "* "
		}
		"dl" {
			set head ""
		}
	}

	foreach itm $itemList {
		if {$kind eq "dl"} {
			lappend block "$prefix[lindex $itm 0]::\n[trac::_wrapListItem [trac::_trimRight [lindex $itm 1]] $prefix]"
		} else {
			lappend block "[trac::_wrapListItem [trac::_trimRight $itm] $prefix $head]"
		} 
	} 
	
	set result [join $block "\n"]
	if {$depth == 1} {
		regsub -all "$marker" $result "" result
	} 
	return $result
}


## 
 # ------------------------------------------------------------------------
 # 
 # "trac::navBarProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::navBarProc {curr prev next top} {
	if {[aida::getParam NavBar]} {
		set result [list]
		if {$curr ne $top} {
			set top "[trac::_pageName $top]"
			lappend result "\[\[$top| [aida::getParam NavTop]\]\]"		
		} 
		if {$prev ne ""} {
			set prev "[trac::_pageName $prev]"
			lappend result "\[\[$prev| [aida::getParam NavPrev]\]\]"
		} 
		if {$next ne ""} {
			set next "[trac::_pageName $next]"
			lappend result "\[\[$next| [aida::getParam NavNext]\]\]"
		} 

		return "[join $result " | "]\[\[br\]\]\n\n"
	} else {
		return ""
	} 
}


## 
 # ------------------------------------------------------------------------
 # 
 # "trac::newLineProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::newLineProc {} {
	return "\[\[br\]\]"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "trac::postambleProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::postambleProc {} {}


## 
 # ------------------------------------------------------------------------
 # 
 # "trac::preambleProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::preambleProc {} {
	return [aida::addPreamble]
}


##
 # ------------------------------------------------------------------------
 # 
 # "trac::printIndexProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::printIndexProc {} {
	variable aida_head

	for {set idx 0} {$idx < [aida::countIndexMarks]} {incr idx} {
		lassign [aida::getIndexMark $idx] fl str
		incr countArr($str)
		set tag "\[\[${fl}#[aida::getParam IndexMark]${idx}| $countArr($str)\]\]"
		lappend indexArr($str) $tag			
	} 
	if {[info exists indexArr]} {
		foreach word [lsort -dict [array names indexArr]] {
			lappend result " * $word\t[join $indexArr($word) ", "]"
		} 
	} 
	return [join $result "\n"]
}


##
 # ------------------------------------------------------------------------
 # 
 # "trac::refProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::refProc {str label {file ""}} {
	set label [string trim $label "\"'"]
	set page [trac::_pageName $file]
	return "\[\[$page#$label| [aida::unwrapText $str]\]\]"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "trac::sectionProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::sectionProc {str level {file ""}} {
	variable aida_head

	set delim [string repeat "=" $level]
	set title [aida::newSectionNumber $level]
	append title [aida::unwrapText [string trim $str]]
	set sc [aida::setSectionMark $level [trac::_pageName $file] $title]
	if {[aida::splitting]} {
		set mark "\[=#[aida::getParam SectionMark]$sc\]\n"
	} else {
		set mark ""
	} 
	
	return "$mark$delim $title $delim"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "trac::setIndexProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::setIndexProc {str {file ""}} {
	variable aida_head
	
	set idx [aida::setIndexMark [trac::_pageName $file] $str]
	set mark "\[=#[aida::getParam IndexMark]$idx\]"
	
	return $mark
}


## 
 # ------------------------------------------------------------------------
 # 
 # "trac::styleProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::styleProc {style begin} {
	switch -- $style {
		"i" {
			set macro "''"
		}
		"b" {
			set macro "'''"
		}
		"u" {
			set macro "__"
		}
		"y" {
			set macro "`"
		}
	}
	return "$macro"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "trac::tableProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::tableProc {attr rowList} {
	set tbl [list]
	set hsep "||"
	
	# Retrieve the attributes
	set attrDict [aida::getAttr table $attr]
	if {[catch {dict get $attrDict border} bord]} {
		set bord 1
	} 

	# In order to have no frame, rely on trac's html wiki processor (see
	# http://trac.edgewall.org/wiki/WikiProcessors) and on Aida's html
	# target
	if {$bord == 0} {
		if {![catch {aida::loadTarget html} res]} {
			lappend tbl "{{{#!html"
			lappend tbl [html::tableProc $attr $rowList] 
			lappend tbl "}}}"
			return [join $tbl "\n"]
		} 
	} 
	
	# Calc the max number of columns
	set ncol 0
	foreach row $rowList {
		set row [split $row "\t"]
		set len [llength $row]
		if {$len > $ncol} {
			set ncol $len 
		} 
	} 

	# Build the format list
	if {[catch {dict get $attrDict format} frmt]} {
		set frmt [string repeat "l" $ncol]
	} 
	regsub -all {\|} $frmt "" frmt
	set posList [split $frmt ""]
	
	# Build the table
	foreach row $rowList {
		set cidx 0
		set line "$hsep"
		foreach cell $row {
			set cell [string trim $cell]
			set pos [lindex $posList $cidx]
			if {$pos eq ""} {set pos "c"} 
			append line "[trac::_alignString $cell $pos]$hsep"
			incr cidx
		} 
		# Complete too short rows
		for {set i $cidx} {$i < $ncol} {incr i} {
			set cell ""
			set pos [lindex $posList $i]
			if {$pos eq ""} {set pos "c"} 
			append line "[trac::_alignString $cell $pos]$hsep"
		}
		lappend tbl $line
	} 
	return [join $tbl "\n"]
}


##
 # ------------------------------------------------------------------------
 # 
 # "trac::tocProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::tocProc {} {
	if {![aida::splitting]} {
		return "\[\[PageOutline(1-3)\]\]"
	} else {
		variable aida_head
			
		set result [list]
		set depth [aida::getParam TocDepth]
		for {set sc 0} {$sc < [aida::countSectionMarks]} {incr sc} {
			lassign [aida::getSectionMark $sc] lv fl title
			if {$lv <= $depth} {
				lappend result "[string repeat " " $lv]* \[\[${fl}#[aida::getParam SectionMark]$sc| $title\]\]"
			} 
		}
		
		return [join $result "\n"]
	} 
	
}


##
 # ------------------------------------------------------------------------
 # 
 # "trac::verbProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::verbProc {str} {
	return "{{{[aida::unwrapText $str]}}}"
}


## 
 # ------------------------------------------------------------------------
 # 
 # "trac::verbatimProc" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::verbatimProc {str} {
	return "{{{\n$str\n}}}"
}



# Target specific utility procs
# =============================


## 
 # ------------------------------------------------------------------------
 # 
 # "trac::_alignString" --
 # 
 # Format a string within a cell. The 'where' argument can be: "l", "r",
 # "c" for left, right or center resp.
 # 
 # Examples:
 # trac::alignString foobar l	-->	/foobar  /
 # trac::alignString foobar r	-->	/  foobar/
 # trac::alignString foobar c	-->	/  foobar  /
 # 
 # ------------------------------------------------------------------------
 ##

proc trac::_alignString {str where} {
	switch -- $where {
		"l" {
			set res "$str  "
		}
		"r" {
			set res "  $str"
		}
		"c" {
			set res "  $str  "
		}
		default {
			error "unknown position specification '$where'"
		}
	}

	return $res
}


## 
 # ------------------------------------------------------------------------
 # 
 # "trac::_wrapListItem" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::_wrapListItem {item prefix {head ""}} {
	set result [list]
	set lines [split $item "\n"]
	set first [lindex $lines 0]
	regexp {^[ \t]*} $first sp
	lappend result "$prefix$head$first"

	for {set i 1} {$i < [llength $lines]} {incr i} {
		regsub {^[ \t]*} [lindex $lines $i] "$prefix$sp" next
		lappend result $next
	}
	
	return [join $result "\n"]
}


## 
 # ------------------------------------------------------------------------
 # 
 # "trac::_trimRight" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::_trimRight {str} {
	return [string trimright $str "\n\r\t "]
}


## 
 # ------------------------------------------------------------------------
 # 
 # "trac::_pageName" --
 # 
 # ------------------------------------------------------------------------
 ##
proc trac::_pageName {file} {
	set ext [aida::getParam NavExtension]
	set result "[file root $file]$ext"
	
	return $result
}

