/*
 * Copyright (C) 2021 Liquidaty and the zsv/lib contributors
 * All rights reserved
 *
 * This file is part of zsv/lib, distributed under the license defined at
 * https://opensource.org/licenses/MIT
 */

#ifndef ZSV_EXT_H
#define ZSV_EXT_H

#include <stdio.h>
#include "common.h"

/**
 * @file ext.h
 * @brief ZSV extension api, common to zsvlib and any zsv extension
 * @defgroup zsv_extension api
 * @ingroup ZSV
 * @{
 */
ZSV_BEGIN_DECL

/**
 * Status and error codes for zsv extensions. If a command's `main()` program
 * returns a status other than zsv_ext_status_ok, `zsv_ext_errcode()` will be
 * called and its result used for the final exit code. If it returns
 * zsv_ext_status_error, then furthermore, `zsv_ext_errstr()` will be called
 * and any non-null result displayed as an error message
 */
enum zsv_ext_status {
  zsv_ext_status_ok = 0,
  zsv_ext_status_memory,
  zsv_ext_status_unrecognized_cmd,

  /**
   * if a zsv_ext function returns zsv_ext_status_error, an error message
   * is generated by calling the module's `zsv_ext_errcode()` and `zsv_ext_errstr()`
   * functions. Otherwise an error string is generated from the zsv_ext_status code
   */
  zsv_ext_status_error,

  /**
   * zsv_ext_status_other is treated as a silent error by ZSV, e.g. if you have
   * already handled your error and do not want any further error message displayed
   */
  zsv_ext_status_other
};

/**
 * The context in which any call to your extension is made, used with:
 *   `set_ext_context()`
 *   `get_ext_context()`
 */
typedef void * zsv_execution_context;

/**
 * Signature of the function called for each implemented sub-command
 */
typedef enum zsv_ext_status (*zsv_ext_main)(zsv_execution_context ctx, int argc, const char *argv[]);

/**
 * ZSV callbacks structure
 *
 * Contains the addresses of the ZSV functions that can be used by custom
 * extensions, and which are passed from ZSV to the extension via the
 * custom `zsv_ext_init()` implementation. See zsv_ext.h for additional
 * extension specifications, or examples/extension.c for an example
 * implementation
 *
 * For a description of each callback, see the corresponding zsv_-prefixed
 * function in zsv.h
 */
struct zsv_ext_callbacks {
  void (*set_row_handler)(zsv_parser handle, void (*row)(void *ctx));
  void (*set_context)(zsv_parser handle, void *ctx);
  enum zsv_status (*parse_more)(struct zsv_scanner *scanner);
  void (*abort)(struct zsv_scanner *scanner);
  size_t (*cell_count)(zsv_parser parser);
  struct zsv_cell (*get_cell)(zsv_parser parser, size_t ix);
  enum zsv_status (*finish)(zsv_parser);
  enum zsv_status (*delete)(zsv_parser);

  /**
   * Set or get custom context (e.g. to persist private data between
   * ZSV-initiated calls to your extension)
   */
  void (*ext_set_context)(zsv_execution_context ctx, void *private_context);
  void *(*ext_get_context)(zsv_execution_context ctx);

  zsv_parser (*ext_get_parser)(zsv_execution_context ctx);

  /**
   * To add an extension command, invoke `ext_add_command`, passing it your command's
   * handler function as a callback with a `zsv_ext_main` signature
   */
  enum zsv_ext_status (*ext_add_command)(zsv_execution_context ctx,
                                         const char *id, const char *help,
                                         zsv_ext_main main);
  void (*ext_set_help)(zsv_execution_context ctx, const char *help);
  void (*ext_set_license)(zsv_execution_context ctx, const char *license);
  void (*ext_set_thirdparty)(zsv_execution_context ctx, const char *thirdparty[]);

  /**
   * fetch options from execution context. used to fetch just the ctx-related parser
   * opts, including any of the below if specified by the user when zsv was invoked:
   *     -B,--buff-size <N>
   *     -c,--max-column-count <N>
   *     -r,--max-row-size <N>
   *     -t,--tab-delim
   *     -O,--other-delim <C>
   *     -q,--no-quote
   *     -v,--verbose
   *
   * @param ctx execution context
   * @return option values
   */
  struct zsv_opts (*ext_parser_opts)(zsv_execution_context ctx);

  /**
   * convenience function that calls ext_args_to_opts, allocates parser,
   * sets custom ctx, runs parser, and de-allocates parser
   */
  enum zsv_ext_status (*ext_parse_all)(zsv_execution_context ctx,
                                       void *user_context,
                                       void (*row_handler)(void *ctx),
                                       struct zsv_opts *const custom);
  /**
   * As an alternative to `ext_parse_all()`, for more granular control:
   * ```
   * struct zsv_opts opts = custom ? *custom : zsv_get_default_opts();
   * zsv_parser parser = zsv_new(&opts);
   * if(!parser)
   *   stat = zsv_ext_status_memory;
   * else {
   *   opts.row = YOUR_COMMAND_rowhandler;
   *   // ... set other options ...
   *   zsv_parser p = new_with_context(ctx, &opts);
   *   while((stat = zsv_parse_more(parser)) == zsv_status_ok) ;
   *   if(stat == zsv_status_no_more_input)
   *     stat = zsv_finish(p);
   *   zsv_delete(p);
   * }
   * ```
   */
};

/** @} */
ZSV_END_DECL
#endif
