import { promises as fs } from 'fs';
import path from 'path';
import { select, confirm } from '@inquirer/prompts';
import { getTaskProgressForChange, formatTaskStatus } from '../utils/task-progress.js';
import { Validator } from './validation/validator.js';
import chalk from 'chalk';
import { extractRequirementsSection, parseDeltaSpec, normalizeRequirementName, } from './parsers/requirement-blocks.js';
export class ArchiveCommand {
    async execute(changeName, options = {}) {
        const targetPath = '.';
        const changesDir = path.join(targetPath, 'openspec', 'changes');
        const archiveDir = path.join(changesDir, 'archive');
        const mainSpecsDir = path.join(targetPath, 'openspec', 'specs');
        // Check if changes directory exists
        try {
            await fs.access(changesDir);
        }
        catch {
            throw new Error("No OpenSpec changes directory found. Run 'openspec init' first.");
        }
        // Get change name interactively if not provided
        if (!changeName) {
            const selectedChange = await this.selectChange(changesDir);
            if (!selectedChange) {
                console.log('No change selected. Aborting.');
                return;
            }
            changeName = selectedChange;
        }
        const changeDir = path.join(changesDir, changeName);
        // Verify change exists
        try {
            const stat = await fs.stat(changeDir);
            if (!stat.isDirectory()) {
                throw new Error(`Change '${changeName}' not found.`);
            }
        }
        catch {
            throw new Error(`Change '${changeName}' not found.`);
        }
        const skipValidation = options.validate === false || options.noValidate === true;
        // Validate specs and change before archiving
        if (!skipValidation) {
            const validator = new Validator();
            let hasValidationErrors = false;
            // Validate proposal.md (non-blocking unless strict mode desired in future)
            const changeFile = path.join(changeDir, 'proposal.md');
            try {
                await fs.access(changeFile);
                const changeReport = await validator.validateChange(changeFile);
                // Proposal validation is informative only (do not block archive)
                if (!changeReport.valid) {
                    console.log(chalk.yellow(`\nProposal warnings in proposal.md (non-blocking):`));
                    for (const issue of changeReport.issues) {
                        const symbol = issue.level === 'ERROR' ? '⚠' : (issue.level === 'WARNING' ? '⚠' : 'ℹ');
                        console.log(chalk.yellow(`  ${symbol} ${issue.message}`));
                    }
                }
            }
            catch {
                // Change file doesn't exist, skip validation
            }
            // Validate delta-formatted spec files under the change directory if present
            const changeSpecsDir = path.join(changeDir, 'specs');
            let hasDeltaSpecs = false;
            try {
                const candidates = await fs.readdir(changeSpecsDir, { withFileTypes: true });
                for (const c of candidates) {
                    if (c.isDirectory()) {
                        try {
                            const candidatePath = path.join(changeSpecsDir, c.name, 'spec.md');
                            await fs.access(candidatePath);
                            const content = await fs.readFile(candidatePath, 'utf-8');
                            if (/^##\s+(ADDED|MODIFIED|REMOVED|RENAMED)\s+Requirements/m.test(content)) {
                                hasDeltaSpecs = true;
                                break;
                            }
                        }
                        catch { }
                    }
                }
            }
            catch { }
            if (hasDeltaSpecs) {
                const deltaReport = await validator.validateChangeDeltaSpecs(changeDir);
                if (!deltaReport.valid) {
                    hasValidationErrors = true;
                    console.log(chalk.red(`\nValidation errors in change delta specs:`));
                    for (const issue of deltaReport.issues) {
                        if (issue.level === 'ERROR') {
                            console.log(chalk.red(`  ✗ ${issue.message}`));
                        }
                        else if (issue.level === 'WARNING') {
                            console.log(chalk.yellow(`  ⚠ ${issue.message}`));
                        }
                    }
                }
            }
            if (hasValidationErrors) {
                console.log(chalk.red('\nValidation failed. Please fix the errors before archiving.'));
                console.log(chalk.yellow('To skip validation (not recommended), use --no-validate flag.'));
                return;
            }
        }
        else {
            // Log warning when validation is skipped
            const timestamp = new Date().toISOString();
            if (!options.yes) {
                const proceed = await confirm({
                    message: chalk.yellow('⚠️  WARNING: Skipping validation may archive invalid specs. Continue? (y/N)'),
                    default: false
                });
                if (!proceed) {
                    console.log('Archive cancelled.');
                    return;
                }
            }
            else {
                console.log(chalk.yellow(`\n⚠️  WARNING: Skipping validation may archive invalid specs.`));
            }
            console.log(chalk.yellow(`[${timestamp}] Validation skipped for change: ${changeName}`));
            console.log(chalk.yellow(`Affected files: ${changeDir}`));
        }
        // Show progress and check for incomplete tasks
        const progress = await getTaskProgressForChange(changesDir, changeName);
        const status = formatTaskStatus(progress);
        console.log(`Task status: ${status}`);
        const incompleteTasks = Math.max(progress.total - progress.completed, 0);
        if (incompleteTasks > 0) {
            if (!options.yes) {
                const proceed = await confirm({
                    message: `Warning: ${incompleteTasks} incomplete task(s) found. Continue?`,
                    default: false
                });
                if (!proceed) {
                    console.log('Archive cancelled.');
                    return;
                }
            }
            else {
                console.log(`Warning: ${incompleteTasks} incomplete task(s) found. Continuing due to --yes flag.`);
            }
        }
        // Handle spec updates unless skipSpecs flag is set
        if (options.skipSpecs) {
            console.log('Skipping spec updates (--skip-specs flag provided).');
        }
        else {
            // Find specs to update
            const specUpdates = await this.findSpecUpdates(changeDir, mainSpecsDir);
            if (specUpdates.length > 0) {
                console.log('\nSpecs to update:');
                for (const update of specUpdates) {
                    const status = update.exists ? 'update' : 'create';
                    const capability = path.basename(path.dirname(update.target));
                    console.log(`  ${capability}: ${status}`);
                }
                let shouldUpdateSpecs = true;
                if (!options.yes) {
                    shouldUpdateSpecs = await confirm({
                        message: 'Proceed with spec updates?',
                        default: true
                    });
                    if (!shouldUpdateSpecs) {
                        console.log('Skipping spec updates. Proceeding with archive.');
                    }
                }
                if (shouldUpdateSpecs) {
                    // Prepare all updates first (validation pass, no writes)
                    const prepared = [];
                    try {
                        for (const update of specUpdates) {
                            const built = await this.buildUpdatedSpec(update, changeName);
                            prepared.push({ update, rebuilt: built.rebuilt, counts: built.counts });
                        }
                    }
                    catch (err) {
                        console.log(String(err.message || err));
                        console.log('Aborted. No files were changed.');
                        return;
                    }
                    // All validations passed; pre-validate rebuilt full spec and then write files and display counts
                    let totals = { added: 0, modified: 0, removed: 0, renamed: 0 };
                    for (const p of prepared) {
                        const specName = path.basename(path.dirname(p.update.target));
                        if (!skipValidation) {
                            const report = await new Validator().validateSpecContent(specName, p.rebuilt);
                            if (!report.valid) {
                                console.log(chalk.red(`\nValidation errors in rebuilt spec for ${specName} (will not write changes):`));
                                for (const issue of report.issues) {
                                    if (issue.level === 'ERROR')
                                        console.log(chalk.red(`  ✗ ${issue.message}`));
                                    else if (issue.level === 'WARNING')
                                        console.log(chalk.yellow(`  ⚠ ${issue.message}`));
                                }
                                console.log('Aborted. No files were changed.');
                                return;
                            }
                        }
                        await this.writeUpdatedSpec(p.update, p.rebuilt, p.counts);
                        totals.added += p.counts.added;
                        totals.modified += p.counts.modified;
                        totals.removed += p.counts.removed;
                        totals.renamed += p.counts.renamed;
                    }
                    console.log(`Totals: + ${totals.added}, ~ ${totals.modified}, - ${totals.removed}, → ${totals.renamed}`);
                    console.log('Specs updated successfully.');
                }
            }
        }
        // Create archive directory with date prefix
        const archiveName = `${this.getArchiveDate()}-${changeName}`;
        const archivePath = path.join(archiveDir, archiveName);
        // Check if archive already exists
        try {
            await fs.access(archivePath);
            throw new Error(`Archive '${archiveName}' already exists.`);
        }
        catch (error) {
            if (error.code !== 'ENOENT') {
                throw error;
            }
        }
        // Create archive directory if needed
        await fs.mkdir(archiveDir, { recursive: true });
        // Move change to archive
        await fs.rename(changeDir, archivePath);
        console.log(`Change '${changeName}' archived as '${archiveName}'.`);
    }
    async selectChange(changesDir) {
        // Get all directories in changes (excluding archive)
        const entries = await fs.readdir(changesDir, { withFileTypes: true });
        const changeDirs = entries
            .filter(entry => entry.isDirectory() && entry.name !== 'archive')
            .map(entry => entry.name)
            .sort();
        if (changeDirs.length === 0) {
            console.log('No active changes found.');
            return null;
        }
        // Build choices with progress inline to avoid duplicate lists
        let choices = changeDirs.map(name => ({ name, value: name }));
        try {
            const progressList = [];
            for (const id of changeDirs) {
                const progress = await getTaskProgressForChange(changesDir, id);
                const status = formatTaskStatus(progress);
                progressList.push({ id, status });
            }
            const nameWidth = Math.max(...progressList.map(p => p.id.length));
            choices = progressList.map(p => ({
                name: `${p.id.padEnd(nameWidth)}     ${p.status}`,
                value: p.id
            }));
        }
        catch {
            // If anything fails, fall back to simple names
            choices = changeDirs.map(name => ({ name, value: name }));
        }
        try {
            const answer = await select({
                message: 'Select a change to archive',
                choices
            });
            return answer;
        }
        catch (error) {
            // User cancelled (Ctrl+C)
            return null;
        }
    }
    // Deprecated: replaced by shared task-progress utilities
    async checkIncompleteTasks(_tasksPath) {
        return 0;
    }
    async findSpecUpdates(changeDir, mainSpecsDir) {
        const updates = [];
        const changeSpecsDir = path.join(changeDir, 'specs');
        try {
            const entries = await fs.readdir(changeSpecsDir, { withFileTypes: true });
            for (const entry of entries) {
                if (entry.isDirectory()) {
                    const specFile = path.join(changeSpecsDir, entry.name, 'spec.md');
                    const targetFile = path.join(mainSpecsDir, entry.name, 'spec.md');
                    try {
                        await fs.access(specFile);
                        // Check if target exists
                        let exists = false;
                        try {
                            await fs.access(targetFile);
                            exists = true;
                        }
                        catch {
                            exists = false;
                        }
                        updates.push({
                            source: specFile,
                            target: targetFile,
                            exists
                        });
                    }
                    catch {
                        // Source spec doesn't exist, skip
                    }
                }
            }
        }
        catch {
            // No specs directory in change
        }
        return updates;
    }
    async buildUpdatedSpec(update, changeName) {
        // Read change spec content (delta-format expected)
        const changeContent = await fs.readFile(update.source, 'utf-8');
        // Parse deltas from the change spec file
        const plan = parseDeltaSpec(changeContent);
        const specName = path.basename(path.dirname(update.target));
        // Pre-validate duplicates within sections
        const addedNames = new Set();
        for (const add of plan.added) {
            const name = normalizeRequirementName(add.name);
            if (addedNames.has(name)) {
                throw new Error(`${specName} validation failed - duplicate requirement in ADDED for header "### Requirement: ${add.name}"`);
            }
            addedNames.add(name);
        }
        const modifiedNames = new Set();
        for (const mod of plan.modified) {
            const name = normalizeRequirementName(mod.name);
            if (modifiedNames.has(name)) {
                throw new Error(`${specName} validation failed - duplicate requirement in MODIFIED for header "### Requirement: ${mod.name}"`);
            }
            modifiedNames.add(name);
        }
        const removedNamesSet = new Set();
        for (const rem of plan.removed) {
            const name = normalizeRequirementName(rem);
            if (removedNamesSet.has(name)) {
                throw new Error(`${specName} validation failed - duplicate requirement in REMOVED for header "### Requirement: ${rem}"`);
            }
            removedNamesSet.add(name);
        }
        const renamedFromSet = new Set();
        const renamedToSet = new Set();
        for (const { from, to } of plan.renamed) {
            const fromNorm = normalizeRequirementName(from);
            const toNorm = normalizeRequirementName(to);
            if (renamedFromSet.has(fromNorm)) {
                throw new Error(`${specName} validation failed - duplicate FROM in RENAMED for header "### Requirement: ${from}"`);
            }
            if (renamedToSet.has(toNorm)) {
                throw new Error(`${specName} validation failed - duplicate TO in RENAMED for header "### Requirement: ${to}"`);
            }
            renamedFromSet.add(fromNorm);
            renamedToSet.add(toNorm);
        }
        // Pre-validate cross-section conflicts
        const conflicts = [];
        for (const n of modifiedNames) {
            if (removedNamesSet.has(n))
                conflicts.push({ name: n, a: 'MODIFIED', b: 'REMOVED' });
            if (addedNames.has(n))
                conflicts.push({ name: n, a: 'MODIFIED', b: 'ADDED' });
        }
        for (const n of addedNames) {
            if (removedNamesSet.has(n))
                conflicts.push({ name: n, a: 'ADDED', b: 'REMOVED' });
        }
        // Renamed interplay: MODIFIED must reference the NEW header, not FROM
        for (const { from, to } of plan.renamed) {
            const fromNorm = normalizeRequirementName(from);
            const toNorm = normalizeRequirementName(to);
            if (modifiedNames.has(fromNorm)) {
                throw new Error(`${specName} validation failed - when a rename exists, MODIFIED must reference the NEW header "### Requirement: ${to}"`);
            }
            // Detect ADDED colliding with a RENAMED TO
            if (addedNames.has(toNorm)) {
                throw new Error(`${specName} validation failed - RENAMED TO header collides with ADDED for "### Requirement: ${to}"`);
            }
        }
        if (conflicts.length > 0) {
            const c = conflicts[0];
            throw new Error(`${specName} validation failed - requirement present in multiple sections (${c.a} and ${c.b}) for header "### Requirement: ${c.name}"`);
        }
        const hasAnyDelta = (plan.added.length + plan.modified.length + plan.removed.length + plan.renamed.length) > 0;
        if (!hasAnyDelta) {
            throw new Error(`Delta parsing found no operations for ${path.basename(path.dirname(update.source))}. ` +
                `Provide ADDED/MODIFIED/REMOVED/RENAMED sections in change spec.`);
        }
        // Load or create base target content
        let targetContent;
        try {
            targetContent = await fs.readFile(update.target, 'utf-8');
        }
        catch {
            // Target spec does not exist; only ADDED operations are permitted
            if (plan.modified.length > 0 || plan.removed.length > 0 || plan.renamed.length > 0) {
                throw new Error(`${specName}: target spec does not exist; only ADDED requirements are allowed for new specs.`);
            }
            targetContent = this.buildSpecSkeleton(specName, changeName);
        }
        // Extract requirements section and build name->block map
        const parts = extractRequirementsSection(targetContent);
        const nameToBlock = new Map();
        for (const block of parts.bodyBlocks) {
            nameToBlock.set(normalizeRequirementName(block.name), block);
        }
        // Apply operations in order: RENAMED → REMOVED → MODIFIED → ADDED
        // RENAMED
        for (const r of plan.renamed) {
            const from = normalizeRequirementName(r.from);
            const to = normalizeRequirementName(r.to);
            if (!nameToBlock.has(from)) {
                throw new Error(`${specName} RENAMED failed for header "### Requirement: ${r.from}" - source not found`);
            }
            if (nameToBlock.has(to)) {
                throw new Error(`${specName} RENAMED failed for header "### Requirement: ${r.to}" - target already exists`);
            }
            const block = nameToBlock.get(from);
            const newHeader = `### Requirement: ${to}`;
            const rawLines = block.raw.split('\n');
            rawLines[0] = newHeader;
            const renamedBlock = {
                headerLine: newHeader,
                name: to,
                raw: rawLines.join('\n'),
            };
            nameToBlock.delete(from);
            nameToBlock.set(to, renamedBlock);
        }
        // REMOVED
        for (const name of plan.removed) {
            const key = normalizeRequirementName(name);
            if (!nameToBlock.has(key)) {
                throw new Error(`${specName} REMOVED failed for header "### Requirement: ${name}" - not found`);
            }
            nameToBlock.delete(key);
        }
        // MODIFIED
        for (const mod of plan.modified) {
            const key = normalizeRequirementName(mod.name);
            if (!nameToBlock.has(key)) {
                throw new Error(`${specName} MODIFIED failed for header "### Requirement: ${mod.name}" - not found`);
            }
            // Replace block with provided raw (ensure header line matches key)
            const modHeaderMatch = mod.raw.split('\n')[0].match(/^###\s*Requirement:\s*(.+)\s*$/);
            if (!modHeaderMatch || normalizeRequirementName(modHeaderMatch[1]) !== key) {
                throw new Error(`${specName} MODIFIED failed for header "### Requirement: ${mod.name}" - header mismatch in content`);
            }
            nameToBlock.set(key, mod);
        }
        // ADDED
        for (const add of plan.added) {
            const key = normalizeRequirementName(add.name);
            if (nameToBlock.has(key)) {
                throw new Error(`${specName} ADDED failed for header "### Requirement: ${add.name}" - already exists`);
            }
            nameToBlock.set(key, add);
        }
        // Duplicates within resulting map are implicitly prevented by key uniqueness.
        // Recompose requirements section preserving original ordering where possible
        const keptOrder = [];
        const seen = new Set();
        for (const block of parts.bodyBlocks) {
            const key = normalizeRequirementName(block.name);
            const replacement = nameToBlock.get(key);
            if (replacement) {
                keptOrder.push(replacement);
                seen.add(key);
            }
        }
        // Append any newly added that were not in original order
        for (const [key, block] of nameToBlock.entries()) {
            if (!seen.has(key)) {
                keptOrder.push(block);
            }
        }
        const reqBody = [
            parts.preamble && parts.preamble.trim() ? parts.preamble.trimEnd() : ''
        ]
            .filter(Boolean)
            .concat(keptOrder.map(b => b.raw))
            .join('\n\n')
            .trimEnd();
        const rebuilt = [
            parts.before.trimEnd(),
            parts.headerLine,
            reqBody,
            parts.after
        ]
            .filter((s, idx) => !(idx === 0 && s === ''))
            .join('\n')
            .replace(/\n{3,}/g, '\n\n');
        return {
            rebuilt,
            counts: {
                added: plan.added.length,
                modified: plan.modified.length,
                removed: plan.removed.length,
                renamed: plan.renamed.length,
            }
        };
    }
    async writeUpdatedSpec(update, rebuilt, counts) {
        // Create target directory if needed
        const targetDir = path.dirname(update.target);
        await fs.mkdir(targetDir, { recursive: true });
        await fs.writeFile(update.target, rebuilt);
        const specName = path.basename(path.dirname(update.target));
        console.log(`Applying changes to openspec/specs/${specName}/spec.md:`);
        if (counts.added)
            console.log(`  + ${counts.added} added`);
        if (counts.modified)
            console.log(`  ~ ${counts.modified} modified`);
        if (counts.removed)
            console.log(`  - ${counts.removed} removed`);
        if (counts.renamed)
            console.log(`  → ${counts.renamed} renamed`);
    }
    buildSpecSkeleton(specFolderName, changeName) {
        const titleBase = specFolderName;
        return `# ${titleBase} Specification\n\n## Purpose\nTBD - created by archiving change ${changeName}. Update Purpose after archive.\n\n## Requirements\n`;
    }
    getArchiveDate() {
        // Returns date in YYYY-MM-DD format
        return new Date().toISOString().split('T')[0];
    }
}
//# sourceMappingURL=archive.js.map