export declare const agentsTemplate = "# OpenSpec Instructions\n\nInstructions for AI coding assistants using OpenSpec for spec-driven development.\n\n## TL;DR Quick Checklist\n\n- Search existing work: `openspec spec list --long`, `openspec list` (use `rg` only for full-text search)\n- Decide scope: new capability vs modify existing capability\n- Pick a unique `change-id`: kebab-case, verb-led (`add-`, `update-`, `remove-`, `refactor-`)\n- Scaffold: `proposal.md`, `tasks.md`, `design.md` (only if needed), and delta specs per affected capability\n- Write deltas: use `## ADDED|MODIFIED|REMOVED|RENAMED Requirements`; include at least one `#### Scenario:` per requirement\n- Validate: `openspec validate [change-id] --strict` and fix issues\n- Request approval: Do not start implementation until proposal is approved\n\n## Three-Stage Workflow\n\n### Stage 1: Creating Changes\nCreate proposal when you need to:\n- Add features or functionality\n- Make breaking changes (API, schema)\n- Change architecture or patterns  \n- Optimize performance (changes behavior)\n- Update security patterns\n\nTriggers (examples):\n- \"Help me create a change proposal\"\n- \"Help me plan a change\"\n- \"Help me create a proposal\"\n- \"I want to create a spec proposal\"\n- \"I want to create a spec\"\n\nLoose matching guidance:\n- Contains one of: `proposal`, `change`, `spec`\n- With one of: `create`, `plan`, `make`, `start`, `help`\n\nSkip proposal for:\n- Bug fixes (restore intended behavior)\n- Typos, formatting, comments\n- Dependency updates (non-breaking)\n- Configuration changes\n- Tests for existing behavior\n\n**Workflow**\n1. Review `openspec/project.md`, `openspec list`, and `openspec list --specs` to understand current context.\n2. Choose a unique verb-led `change-id` and scaffold `proposal.md`, `tasks.md`, optional `design.md`, and spec deltas under `openspec/changes/<id>/`.\n3. Draft spec deltas using `## ADDED|MODIFIED|REMOVED Requirements` with at least one `#### Scenario:` per requirement.\n4. Run `openspec validate <id> --strict` and resolve any issues before sharing the proposal.\n\n### Stage 2: Implementing Changes\nTrack these steps as TODOs and complete them one by one.\n1. **Read proposal.md** - Understand what's being built\n2. **Read design.md** (if exists) - Review technical decisions\n3. **Read tasks.md** - Get implementation checklist\n4. **Implement tasks sequentially** - Complete in order\n5. **Confirm completion** - Ensure every item in `tasks.md` is finished before updating statuses\n6. **Update checklist** - After all work is done, set every task to `- [x]` so the list reflects reality\n7. **Approval gate** - Do not start implementation until the proposal is reviewed and approved\n\n### Stage 3: Archiving Changes\nAfter deployment, create separate PR to:\n- Move `changes/[name]/` \u2192 `changes/archive/YYYY-MM-DD-[name]/`\n- Update `specs/` if capabilities changed\n- Use `openspec archive <change-id> --skip-specs --yes` for tooling-only changes (always pass the change ID explicitly)\n- Run `openspec validate --strict` to confirm the archived change passes checks\n\n## Before Any Task\n\n**Context Checklist:**\n- [ ] Read relevant specs in `specs/[capability]/spec.md`\n- [ ] Check pending changes in `changes/` for conflicts\n- [ ] Read `openspec/project.md` for conventions\n- [ ] Run `openspec list` to see active changes\n- [ ] Run `openspec list --specs` to see existing capabilities\n\n**Before Creating Specs:**\n- Always check if capability already exists\n- Prefer modifying existing specs over creating duplicates\n- Use `openspec show [spec]` to review current state\n- If request is ambiguous, ask 1\u20132 clarifying questions before scaffolding\n\n### Search Guidance\n- Enumerate specs: `openspec spec list --long` (or `--json` for scripts)\n- Enumerate changes: `openspec list` (or `openspec change list --json` - deprecated but available)\n- Show details:\n  - Spec: `openspec show <spec-id> --type spec` (use `--json` for filters)\n  - Change: `openspec show <change-id> --json --deltas-only`\n- Full-text search (use ripgrep): `rg -n \"Requirement:|Scenario:\" openspec/specs`\n\n## Quick Start\n\n### CLI Commands\n\n```bash\n# Essential commands\nopenspec list                  # List active changes\nopenspec list --specs          # List specifications\nopenspec show [item]           # Display change or spec\nopenspec validate [item]       # Validate changes or specs\nopenspec archive <change-id> [--yes|-y]   # Archive after deployment (add --yes for non-interactive runs)\n\n# Project management\nopenspec init [path]           # Initialize OpenSpec\nopenspec update [path]         # Update instruction files\n\n# Interactive mode\nopenspec show                  # Prompts for selection\nopenspec validate              # Bulk validation mode\n\n# Debugging\nopenspec show [change] --json --deltas-only\nopenspec validate [change] --strict\n```\n\n### Command Flags\n\n- `--json` - Machine-readable output\n- `--type change|spec` - Disambiguate items\n- `--strict` - Comprehensive validation\n- `--no-interactive` - Disable prompts\n- `--skip-specs` - Archive without spec updates\n- `--yes`/`-y` - Skip confirmation prompts (non-interactive archive)\n\n## Directory Structure\n\n```\nopenspec/\n\u251C\u2500\u2500 project.md              # Project conventions\n\u251C\u2500\u2500 specs/                  # Current truth - what IS built\n\u2502   \u2514\u2500\u2500 [capability]/       # Single focused capability\n\u2502       \u251C\u2500\u2500 spec.md         # Requirements and scenarios\n\u2502       \u2514\u2500\u2500 design.md       # Technical patterns\n\u251C\u2500\u2500 changes/                # Proposals - what SHOULD change\n\u2502   \u251C\u2500\u2500 [change-name]/\n\u2502   \u2502   \u251C\u2500\u2500 proposal.md     # Why, what, impact\n\u2502   \u2502   \u251C\u2500\u2500 tasks.md        # Implementation checklist\n\u2502   \u2502   \u251C\u2500\u2500 design.md       # Technical decisions (optional; see criteria)\n\u2502   \u2502   \u2514\u2500\u2500 specs/          # Delta changes\n\u2502   \u2502       \u2514\u2500\u2500 [capability]/\n\u2502   \u2502           \u2514\u2500\u2500 spec.md # ADDED/MODIFIED/REMOVED\n\u2502   \u2514\u2500\u2500 archive/            # Completed changes\n```\n\n## Creating Change Proposals\n\n### Decision Tree\n\n```\nNew request?\n\u251C\u2500 Bug fix restoring spec behavior? \u2192 Fix directly\n\u251C\u2500 Typo/format/comment? \u2192 Fix directly  \n\u251C\u2500 New feature/capability? \u2192 Create proposal\n\u251C\u2500 Breaking change? \u2192 Create proposal\n\u251C\u2500 Architecture change? \u2192 Create proposal\n\u2514\u2500 Unclear? \u2192 Create proposal (safer)\n```\n\n### Proposal Structure\n\n1. **Create directory:** `changes/[change-id]/` (kebab-case, verb-led, unique)\n\n2. **Write proposal.md:**\n```markdown\n# Change: [Brief description of change]\n\n## Why\n[1-2 sentences on problem/opportunity]\n\n## What Changes\n- [Bullet list of changes]\n- [Mark breaking changes with **BREAKING**]\n\n## Impact\n- Affected specs: [list capabilities]\n- Affected code: [key files/systems]\n```\n\n3. **Create spec deltas:** `specs/[capability]/spec.md`\n```markdown\n## ADDED Requirements\n### Requirement: New Feature\nThe system SHALL provide...\n\n#### Scenario: Success case\n- **WHEN** user performs action\n- **THEN** expected result\n\n## MODIFIED Requirements\n### Requirement: Existing Feature\n[Complete modified requirement]\n\n## REMOVED Requirements\n### Requirement: Old Feature\n**Reason**: [Why removing]\n**Migration**: [How to handle]\n```\nIf multiple capabilities are affected, create multiple delta files under `changes/[change-id]/specs/<capability>/spec.md`\u2014one per capability.\n\n4. **Create tasks.md:**\n```markdown\n## 1. Implementation\n- [ ] 1.1 Create database schema\n- [ ] 1.2 Implement API endpoint\n- [ ] 1.3 Add frontend component\n- [ ] 1.4 Write tests\n```\n\n5. **Create design.md when needed:**\nCreate `design.md` if any of the following apply; otherwise omit it:\n- Cross-cutting change (multiple services/modules) or a new architectural pattern\n- New external dependency or significant data model changes\n- Security, performance, or migration complexity\n- Ambiguity that benefits from technical decisions before coding\n\nMinimal `design.md` skeleton:\n```markdown\n## Context\n[Background, constraints, stakeholders]\n\n## Goals / Non-Goals\n- Goals: [...]\n- Non-Goals: [...]\n\n## Decisions\n- Decision: [What and why]\n- Alternatives considered: [Options + rationale]\n\n## Risks / Trade-offs\n- [Risk] \u2192 Mitigation\n\n## Migration Plan\n[Steps, rollback]\n\n## Open Questions\n- [...]\n```\n\n## Spec File Format\n\n### Critical: Scenario Formatting\n\n**CORRECT** (use #### headers):\n```markdown\n#### Scenario: User login success\n- **WHEN** valid credentials provided\n- **THEN** return JWT token\n```\n\n**WRONG** (don't use bullets or bold):\n```markdown\n- **Scenario: User login**  \u274C\n**Scenario**: User login     \u274C\n### Scenario: User login      \u274C\n```\n\nEvery requirement MUST have at least one scenario.\n\n### Requirement Wording\n- Use SHALL/MUST for normative requirements (avoid should/may unless intentionally non-normative)\n\n### Delta Operations\n\n- `## ADDED Requirements` - New capabilities\n- `## MODIFIED Requirements` - Changed behavior\n- `## REMOVED Requirements` - Deprecated features\n- `## RENAMED Requirements` - Name changes\n\nHeaders matched with `trim(header)` - whitespace ignored.\n\n#### When to use ADDED vs MODIFIED\n- ADDED: Introduces a new capability or sub-capability that can stand alone as a requirement. Prefer ADDED when the change is orthogonal (e.g., adding \"Slash Command Configuration\") rather than altering the semantics of an existing requirement.\n- MODIFIED: Changes the behavior, scope, or acceptance criteria of an existing requirement. Always paste the full, updated requirement content (header + all scenarios). The archiver will replace the entire requirement with what you provide here; partial deltas will drop previous details.\n- RENAMED: Use when only the name changes. If you also change behavior, use RENAMED (name) plus MODIFIED (content) referencing the new name.\n\nCommon pitfall: Using MODIFIED to add a new concern without including the previous text. This causes loss of detail at archive time. If you aren\u2019t explicitly changing the existing requirement, add a new requirement under ADDED instead.\n\nAuthoring a MODIFIED requirement correctly:\n1) Locate the existing requirement in `openspec/specs/<capability>/spec.md`.\n2) Copy the entire requirement block (from `### Requirement: ...` through its scenarios).\n3) Paste it under `## MODIFIED Requirements` and edit to reflect the new behavior.\n4) Ensure the header text matches exactly (whitespace-insensitive) and keep at least one `#### Scenario:`.\n\nExample for RENAMED:\n```markdown\n## RENAMED Requirements\n- FROM: `### Requirement: Login`\n- TO: `### Requirement: User Authentication`\n```\n\n## Troubleshooting\n\n### Common Errors\n\n**\"Change must have at least one delta\"**\n- Check `changes/[name]/specs/` exists with .md files\n- Verify files have operation prefixes (## ADDED Requirements)\n\n**\"Requirement must have at least one scenario\"**\n- Check scenarios use `#### Scenario:` format (4 hashtags)\n- Don't use bullet points or bold for scenario headers\n\n**Silent scenario parsing failures**\n- Exact format required: `#### Scenario: Name`\n- Debug with: `openspec show [change] --json --deltas-only`\n\n### Validation Tips\n\n```bash\n# Always use strict mode for comprehensive checks\nopenspec validate [change] --strict\n\n# Debug delta parsing\nopenspec show [change] --json | jq '.deltas'\n\n# Check specific requirement\nopenspec show [spec] --json -r 1\n```\n\n## Happy Path Script\n\n```bash\n# 1) Explore current state\nopenspec spec list --long\nopenspec list\n# Optional full-text search:\n# rg -n \"Requirement:|Scenario:\" openspec/specs\n# rg -n \"^#|Requirement:\" openspec/changes\n\n# 2) Choose change id and scaffold\nCHANGE=add-two-factor-auth\nmkdir -p openspec/changes/$CHANGE/{specs/auth}\nprintf \"## Why\\n...\\n\\n## What Changes\\n- ...\\n\\n## Impact\\n- ...\\n\" > openspec/changes/$CHANGE/proposal.md\nprintf \"## 1. Implementation\\n- [ ] 1.1 ...\\n\" > openspec/changes/$CHANGE/tasks.md\n\n# 3) Add deltas (example)\ncat > openspec/changes/$CHANGE/specs/auth/spec.md << 'EOF'\n## ADDED Requirements\n### Requirement: Two-Factor Authentication\nUsers MUST provide a second factor during login.\n\n#### Scenario: OTP required\n- **WHEN** valid credentials are provided\n- **THEN** an OTP challenge is required\nEOF\n\n# 4) Validate\nopenspec validate $CHANGE --strict\n```\n\n## Multi-Capability Example\n\n```\nopenspec/changes/add-2fa-notify/\n\u251C\u2500\u2500 proposal.md\n\u251C\u2500\u2500 tasks.md\n\u2514\u2500\u2500 specs/\n    \u251C\u2500\u2500 auth/\n    \u2502   \u2514\u2500\u2500 spec.md   # ADDED: Two-Factor Authentication\n    \u2514\u2500\u2500 notifications/\n        \u2514\u2500\u2500 spec.md   # ADDED: OTP email notification\n```\n\nauth/spec.md\n```markdown\n## ADDED Requirements\n### Requirement: Two-Factor Authentication\n...\n```\n\nnotifications/spec.md\n```markdown\n## ADDED Requirements\n### Requirement: OTP Email Notification\n...\n```\n\n## Best Practices\n\n### Simplicity First\n- Default to <100 lines of new code\n- Single-file implementations until proven insufficient\n- Avoid frameworks without clear justification\n- Choose boring, proven patterns\n\n### Complexity Triggers\nOnly add complexity with:\n- Performance data showing current solution too slow\n- Concrete scale requirements (>1000 users, >100MB data)\n- Multiple proven use cases requiring abstraction\n\n### Clear References\n- Use `file.ts:42` format for code locations\n- Reference specs as `specs/auth/spec.md`\n- Link related changes and PRs\n\n### Capability Naming\n- Use verb-noun: `user-auth`, `payment-capture`\n- Single purpose per capability\n- 10-minute understandability rule\n- Split if description needs \"AND\"\n\n### Change ID Naming\n- Use kebab-case, short and descriptive: `add-two-factor-auth`\n- Prefer verb-led prefixes: `add-`, `update-`, `remove-`, `refactor-`\n- Ensure uniqueness; if taken, append `-2`, `-3`, etc.\n\n## Tool Selection Guide\n\n| Task | Tool | Why |\n|------|------|-----|\n| Find files by pattern | Glob | Fast pattern matching |\n| Search code content | Grep | Optimized regex search |\n| Read specific files | Read | Direct file access |\n| Explore unknown scope | Task | Multi-step investigation |\n\n## Error Recovery\n\n### Change Conflicts\n1. Run `openspec list` to see active changes\n2. Check for overlapping specs\n3. Coordinate with change owners\n4. Consider combining proposals\n\n### Validation Failures\n1. Run with `--strict` flag\n2. Check JSON output for details\n3. Verify spec file format\n4. Ensure scenarios properly formatted\n\n### Missing Context\n1. Read project.md first\n2. Check related specs\n3. Review recent archives\n4. Ask for clarification\n\n## Quick Reference\n\n### Stage Indicators\n- `changes/` - Proposed, not yet built\n- `specs/` - Built and deployed\n- `archive/` - Completed changes\n\n### File Purposes\n- `proposal.md` - Why and what\n- `tasks.md` - Implementation steps\n- `design.md` - Technical decisions\n- `spec.md` - Requirements and behavior\n\n### CLI Essentials\n```bash\nopenspec list              # What's in progress?\nopenspec show [item]       # View details\nopenspec validate --strict # Is it correct?\nopenspec archive <change-id> [--yes|-y]  # Mark complete (add --yes for automation)\n```\n\nRemember: Specs are truth. Changes are proposals. Keep them in sync.\n";
//# sourceMappingURL=agents-template.d.ts.map