import { z } from 'zod';
// Artifact definition schema
export const ArtifactSchema = z.object({
    id: z.string().min(1, { error: 'Artifact ID is required' }),
    generates: z.string().min(1, { error: 'generates field is required' }),
    description: z.string(),
    template: z.string().min(1, { error: 'template field is required' }),
    instruction: z.string().optional(),
    requires: z.array(z.string()).default([]),
});
// Apply phase configuration for schema-aware apply instructions
export const ApplyPhaseSchema = z.object({
    // Artifact IDs that must exist before apply is available
    requires: z.array(z.string()).min(1, { error: 'At least one required artifact' }),
    // Path to file with checkboxes for progress (relative to change dir), or null if no tracking
    tracks: z.string().nullable().optional(),
    // Custom guidance for the apply phase
    instruction: z.string().optional(),
});
// Full schema YAML structure
export const SchemaYamlSchema = z.object({
    name: z.string().min(1, { error: 'Schema name is required' }),
    version: z.number().int().positive({ error: 'Version must be a positive integer' }),
    description: z.string().optional(),
    artifacts: z.array(ArtifactSchema).min(1, { error: 'At least one artifact required' }),
    // Optional apply phase configuration (for schema-aware apply instructions)
    apply: ApplyPhaseSchema.optional(),
});
// Per-change metadata schema
// Note: schema field is validated at parse time against available schemas
// using a lazy import to avoid circular dependencies
export const ChangeMetadataSchema = z.object({
    // Required: which workflow schema this change uses
    schema: z.string().min(1, { message: 'schema is required' }),
    // Optional: creation timestamp (ISO date string)
    created: z
        .string()
        .regex(/^\d{4}-\d{2}-\d{2}$/, {
        message: 'created must be YYYY-MM-DD format',
    })
        .optional(),
});
//# sourceMappingURL=types.js.map