from collections.abc import Iterable, Iterator, Sequence
import enum
from typing import Annotated, overload

import numpy
from numpy.typing import NDArray

from . import cif as cif


class CrystalSystem(enum.Enum):
    Triclinic = 0

    Monoclinic = 1

    Orthorhombic = 2

    Tetragonal = 3

    Trigonal = 4

    Hexagonal = 5

    Cubic = 6

class Op:
    @overload
    def __init__(self) -> None: ...

    @overload
    def __init__(self, arg: str, /) -> None: ...

    DEN: int = ...
    """Denominator (integer) for the translation vector."""

    @property
    def rot(self) -> list[list[int]]:
        """3x3 integer matrix."""

    @rot.setter
    def rot(self, arg: Sequence[Sequence[int]], /) -> None: ...

    @property
    def tran(self) -> list[int]:
        """Numerators (integers) of the translation vector. Denominator DEN=24."""

    @tran.setter
    def tran(self, arg: Sequence[int], /) -> None: ...

    def is_hkl(self) -> bool: ...

    def as_hkl(self) -> Op: ...

    def as_xyz(self) -> Op: ...

    def triplet(self, style: str = ' ') -> str: ...

    def inverse(self) -> Op:
        """Returns inverted operator."""

    def wrap(self) -> Op:
        """Wrap the translation part to [0,1)"""

    def translated(self, a: Sequence[int]) -> Op:
        """Adds a to tran"""

    def transposed_rot(self) -> list[list[int]]: ...

    def det_rot(self) -> int:
        """Determinant of the 3x3 matrix."""

    def rot_type(self) -> int: ...

    def combine(self, b: Op) -> Op:
        """Combine two symmetry operations."""

    def seitz(self) -> list:
        """Returns Seitz matrix (fractions)"""

    def float_seitz(self) -> list[list[float]]:
        """Returns Seitz matrix (floats)"""

    def apply_to_xyz(self, xyz: Sequence[float]) -> list[float]: ...

    def apply_to_hkl(self, hkl: Sequence[int]) -> list[int]: ...

    def phase_shift(self, hkl: Sequence[int]) -> float: ...

    @overload
    def __mul__(self, arg: Op, /) -> Op: ...

    @overload
    def __mul__(self, arg: str, /) -> Op: ...

    def __rmul__(self, arg: str, /) -> Op: ...

    @overload
    def __eq__(self, arg: Op, /) -> bool: ...

    @overload
    def __eq__(self, arg: object, /) -> bool: ...

    def __copy__(self) -> Op: ...

    def __deepcopy__(self, memo: dict) -> Op: ...

    def __hash__(self) -> int: ...

    def __repr__(self) -> str: ...

def parse_triplet(triplet: str, notation: str = ' ') -> Op:
    """Parse coordinate triplet into gemmi.Op."""

def parse_triplet_part(arg: str, /) -> list[int]: ...

class GroupOps:
    def __init__(self, arg: Sequence[Op], /) -> None: ...

    def __iter__(self) -> Iterator[Op]: ...

    def __eq__(self, arg: object, /) -> bool: ...

    def __len__(self) -> int: ...

    def __deepcopy__(self, memo: dict) -> GroupOps: ...

    @property
    def sym_ops(self) -> list[Op]:
        """Symmetry operations (to be combined with centering vectors)."""

    @sym_ops.setter
    def sym_ops(self, arg: Sequence[Op], /) -> None: ...

    @property
    def cen_ops(self) -> list[list[int]]:
        """Centering vectors."""

    @cen_ops.setter
    def cen_ops(self, arg: Sequence[Sequence[int]], /) -> None: ...

    def add_missing_elements(self) -> None: ...

    def find_centering(self) -> str: ...

    def has_same_centring(self, arg: GroupOps, /) -> bool: ...

    def has_same_rotations(self, arg: GroupOps, /) -> bool: ...

    def is_centrosymmetric(self) -> bool: ...

    def is_reflection_centric(self, arg: Sequence[int], /) -> bool: ...

    def centric_flag_array(self, arg: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), device='cpu')], /) -> NDArray[numpy.bool]: ...

    def epsilon_factor_without_centering(self, arg: Sequence[int], /) -> int: ...

    def epsilon_factor(self, arg: Sequence[int], /) -> int: ...

    def epsilon_factor_array(self, arg: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), device='cpu')], /) -> NDArray[numpy.int32]: ...

    def epsilon_factor_without_centering_array(self, arg: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), device='cpu')], /) -> NDArray[numpy.int32]: ...

    def is_systematically_absent(self, arg: Sequence[int], /) -> bool: ...

    def systematic_absences(self, arg: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), device='cpu')], /) -> NDArray[numpy.bool]: ...

    def find_grid_factors(self) -> list[int]:
        """Minimal multiplicity for real-space grid (e.g. 1,1,6 for P61)."""

    def change_basis_forward(self, cob: Op) -> None:
        """Applies the change-of-basis operator (in place)."""

    def change_basis_backward(self, cob: Op) -> None:
        """Applies inverse of the change-of-basis operator (in place)."""

    def derive_symmorphic(self) -> GroupOps: ...

    def add_inversion(self) -> bool: ...

class SpaceGroup:
    @overload
    def __init__(self, ccp4: int) -> None: ...

    @overload
    def __init__(self, hm: str) -> None: ...

    @property
    def number(self) -> int:
        """number 1-230."""

    @property
    def ccp4(self) -> int:
        """ccp4 number"""

    @property
    def hm(self) -> str:
        """Hermann-Mauguin name"""

    @property
    def ext(self) -> str:
        """Extension (1, 2, H, R or none)"""

    @property
    def qualifier(self) -> str:
        """e.g. 'cab'"""

    @property
    def hall(self) -> str:
        """Hall symbol"""

    @property
    def basisop(self) -> Op: ...

    def xhm(self) -> str:
        """extended Hermann-Mauguin name"""

    def centring_type(self) -> str: ...

    def short_name(self) -> str:
        """H-M name w/o spaces and with 1's removed in '1 ... 1'."""

    def is_enantiomorphic(self) -> bool: ...

    def is_sohncke(self) -> bool: ...

    def is_symmorphic(self) -> bool: ...

    def point_group_hm(self) -> str:
        """Returns H-M name of the point group."""

    def laue_str(self) -> str:
        """
        Returns name of the Laue class (for centrosymmetric groups the same as point_group_hm).
        """

    def crystal_system(self) -> CrystalSystem: ...

    def crystal_system_str(self) -> str:
        """Returns lower-case name of the crystal system."""

    def is_centrosymmetric(self) -> bool: ...

    def monoclinic_unique_axis(self) -> str: ...

    def is_reference_setting(self) -> bool: ...

    def centred_to_primitive(self) -> Op: ...

    def change_of_hand_op(self) -> Op: ...

    def operations(self) -> GroupOps:
        """Group of operations"""

    def switch_to_asu(self, miller_array: Annotated[NDArray[numpy.int32], dict(shape=(None, None), device='cpu')]) -> None: ...

    def __eq__(self, arg: object, /) -> bool: ...

    def __reduce__(self) -> tuple: ...

    def __repr__(self) -> str: ...

class ReciprocalAsu:
    def __init__(self, arg0: SpaceGroup, tnt: bool = False) -> None: ...

    def is_in(self, hkl: Sequence[int]) -> bool: ...

    def condition_str(self) -> str: ...

    def to_asu(self, hkl: Sequence[int], group_ops: GroupOps) -> tuple[list[int], int]: ...

def spacegroup_table() -> Iterator[SpaceGroup]: ...

def spacegroup_table_itb() -> Iterator[SpaceGroup]: ...

def generators_from_hall(hall: str) -> GroupOps:
    """Parse Hall notation."""

def symops_from_hall(hall: str) -> GroupOps:
    """Parse Hall notation."""

def find_spacegroup_by_number(ccp4: int) -> SpaceGroup:
    """Returns space-group of given number."""

def find_spacegroup_by_name(hm: str, alpha: float = 0.0, gamma: float = 0.0, prefer: str = '') -> SpaceGroup:
    """Returns space-group with given name."""

def get_spacegroup_reference_setting(number: int) -> SpaceGroup: ...

def find_spacegroup_by_ops(group_ops: GroupOps) -> SpaceGroup:
    """Returns space-group with identical operations."""

def seitz_to_op(arg: Sequence[Sequence[float]], /) -> Op: ...

class Vec3:
    def __init__(self, arg0: float, arg1: float, arg2: float, /) -> None: ...

    @property
    def x(self) -> float: ...

    @x.setter
    def x(self, arg: float, /) -> None: ...

    @property
    def y(self) -> float: ...

    @y.setter
    def y(self, arg: float, /) -> None: ...

    @property
    def z(self) -> float: ...

    @z.setter
    def z(self, arg: float, /) -> None: ...

    def normalized(self) -> Vec3: ...

    def dot(self, arg: Vec3, /) -> float: ...

    def cross(self, arg: Vec3, /) -> Vec3: ...

    def length(self) -> float: ...

    def approx(self, other: Vec3, epsilon: float) -> bool: ...

    def tolist(self) -> list[float]: ...

    def fromlist(self, arg: Sequence[float], /) -> None: ...

    def __add__(self, arg: Vec3, /) -> Vec3: ...

    def __sub__(self, arg: Vec3, /) -> Vec3: ...

    def __iadd__(self, arg: Vec3, /) -> Vec3: ...

    def __isub__(self, arg: Vec3, /) -> Vec3: ...

    def __mul__(self, arg: float, /) -> Vec3: ...

    def __imul__(self, arg: float, /) -> Vec3: ...

    def __rmul__(self, arg: float, /) -> Vec3: ...

    def __truediv__(self, arg: float, /) -> Vec3: ...

    def __itruediv__(self, arg: float, /) -> Vec3: ...

    def __neg__(self) -> Vec3: ...

    def __getitem__(self, arg: int, /) -> float: ...

    def __setitem__(self, arg0: int, arg1: float, /) -> None: ...

    def __repr__(self) -> str: ...

class Mat33:
    @overload
    def __init__(self) -> None: ...

    @overload
    def __init__(self, arg: Sequence[Sequence[float]], /) -> None: ...

    @property
    def array(self) -> Annotated[NDArray[numpy.float64], dict(shape=(3, 3), order='C')]: ...

    def __array__(self, dtype: object | None = None, copy: object | None = None) -> object: ...

    def row_copy(self, arg: int, /) -> Vec3: ...

    def column_copy(self, arg: int, /) -> Vec3: ...

    def __add__(self, arg: Mat33, /) -> Mat33: ...

    def __sub__(self, arg: Mat33, /) -> Mat33: ...

    @overload
    def multiply(self, arg: Mat33, /) -> Mat33: ...

    @overload
    def multiply(self, arg: Vec3, /) -> Vec3: ...

    @overload
    def __matmul__(self, arg: Mat33, /) -> Mat33: ...

    @overload
    def __matmul__(self, arg: Vec3, /) -> Vec3: ...

    def left_multiply(self, arg: Vec3, /) -> Vec3: ...

    def multiply_by_diagonal(self, arg: Vec3, /) -> Mat33: ...

    def transpose(self) -> Mat33: ...

    def trace(self) -> float: ...

    def approx(self, other: Mat33, epsilon: float) -> bool: ...

    def determinant(self) -> float: ...

    def inverse(self) -> Mat33: ...

    def is_identity(self) -> bool: ...

    def tolist(self) -> list[list[float]]: ...

    def fromlist(self, arg: Sequence[Sequence[float]], /) -> None: ...

    def __repr__(self) -> str: ...

class SMat33d:
    def __init__(self, u11: float, u22: float, u33: float, u12: float, u13: float, u23: float) -> None: ...

    @property
    def u11(self) -> float: ...

    @u11.setter
    def u11(self, arg: float, /) -> None: ...

    @property
    def u22(self) -> float: ...

    @u22.setter
    def u22(self, arg: float, /) -> None: ...

    @property
    def u33(self) -> float: ...

    @u33.setter
    def u33(self, arg: float, /) -> None: ...

    @property
    def u12(self) -> float: ...

    @u12.setter
    def u12(self, arg: float, /) -> None: ...

    @property
    def u13(self) -> float: ...

    @u13.setter
    def u13(self, arg: float, /) -> None: ...

    @property
    def u23(self) -> float: ...

    @u23.setter
    def u23(self, arg: float, /) -> None: ...

    def elements_pdb(self) -> list[float]: ...

    def elements_voigt(self) -> list[float]: ...

    def as_mat33(self) -> Mat33: ...

    def trace(self) -> float: ...

    def nonzero(self) -> bool: ...

    def determinant(self) -> float: ...

    def inverse(self) -> SMat33d: ...

    def scaled(self, arg: float, /) -> SMat33d: ...

    def added_kI(self, arg: float, /) -> SMat33d: ...

    @overload
    def r_u_r(self, arg: Vec3, /) -> float: ...

    @overload
    def r_u_r(self, arg: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), device='cpu')]) -> Annotated[NDArray[numpy.float64], dict(shape=(None,))]: ...

    def multiply(self, arg: Vec3, /) -> Vec3: ...

    def __add__(self, arg: SMat33d, /) -> SMat33d: ...

    def __sub__(self, arg: SMat33d, /) -> SMat33d: ...

    def transformed_by(self, arg: Mat33, /) -> SMat33d: ...

    def calculate_eigenvalues(self) -> list[float]: ...

    def __repr__(self) -> str: ...

class SMat33f:
    def __init__(self, u11: float, u22: float, u33: float, u12: float, u13: float, u23: float) -> None: ...

    @property
    def u11(self) -> float: ...

    @u11.setter
    def u11(self, arg: float, /) -> None: ...

    @property
    def u22(self) -> float: ...

    @u22.setter
    def u22(self, arg: float, /) -> None: ...

    @property
    def u33(self) -> float: ...

    @u33.setter
    def u33(self, arg: float, /) -> None: ...

    @property
    def u12(self) -> float: ...

    @u12.setter
    def u12(self, arg: float, /) -> None: ...

    @property
    def u13(self) -> float: ...

    @u13.setter
    def u13(self, arg: float, /) -> None: ...

    @property
    def u23(self) -> float: ...

    @u23.setter
    def u23(self, arg: float, /) -> None: ...

    def elements_pdb(self) -> list[float]: ...

    def elements_voigt(self) -> list[float]: ...

    def as_mat33(self) -> Mat33: ...

    def trace(self) -> float: ...

    def nonzero(self) -> bool: ...

    def determinant(self) -> float: ...

    def inverse(self) -> SMat33f: ...

    def scaled(self, arg: float, /) -> SMat33f: ...

    def added_kI(self, arg: float, /) -> SMat33f: ...

    @overload
    def r_u_r(self, arg: Vec3, /) -> float: ...

    @overload
    def r_u_r(self, arg: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), device='cpu')]) -> Annotated[NDArray[numpy.float32], dict(shape=(None,))]: ...

    def multiply(self, arg: Vec3, /) -> Vec3: ...

    def __add__(self, arg: SMat33f, /) -> SMat33f: ...

    def __sub__(self, arg: SMat33f, /) -> SMat33f: ...

    def transformed_by(self, arg: Mat33, /) -> SMat33f: ...

    def calculate_eigenvalues(self) -> list[float]: ...

    def __repr__(self) -> str: ...

class Transform:
    @overload
    def __init__(self) -> None: ...

    @overload
    def __init__(self, mat33: Mat33, vec3: Vec3) -> None: ...

    @property
    def mat(self) -> Mat33: ...

    @property
    def vec(self) -> Vec3: ...

    def inverse(self) -> Transform: ...

    def apply(self, arg: Vec3, /) -> Vec3: ...

    def combine(self, arg: Transform, /) -> Transform: ...

    def __matmul__(self, arg: Transform, /) -> Transform: ...

    def is_identity(self) -> bool: ...

    def approx(self, other: Transform, epsilon: float) -> bool: ...

class Position(Vec3):
    @overload
    def __init__(self, arg0: float, arg1: float, arg2: float, /) -> None: ...

    @overload
    def __init__(self, arg: Vec3, /) -> None: ...

    def dist(self, arg: Position, /) -> float: ...

    def __add__(self, arg: Position, /) -> Position: ...  # type: ignore[override]

    def __sub__(self, arg: Position, /) -> Position: ...  # type: ignore[override]

    def __iadd__(self, arg: Position, /) -> Position: ...  # type: ignore[override]

    def __isub__(self, arg: Position, /) -> Position: ...  # type: ignore[override]

    def __mul__(self, arg: float, /) -> Position: ...

    def __imul__(self, arg: float, /) -> Position: ...

    def __rmul__(self, arg: float, /) -> Position: ...

    def __truediv__(self, arg: float, /) -> Position: ...

    def __itruediv__(self, arg: float, /) -> Position: ...

    def __neg__(self) -> Position: ...

    def __repr__(self) -> str: ...

class Fractional(Vec3):
    @overload
    def __init__(self, arg0: float, arg1: float, arg2: float, /) -> None: ...

    @overload
    def __init__(self, arg: Vec3, /) -> None: ...

    def wrap_to_unit(self) -> Fractional: ...

    def wrap_to_zero(self) -> Fractional: ...

    def __getitem__(self, arg: int, /) -> float: ...

    def __add__(self, arg: Fractional, /) -> Fractional: ...  # type: ignore[override]

    def __sub__(self, arg: Fractional, /) -> Fractional: ...  # type: ignore[override]

    def __repr__(self) -> str: ...

class PositionBox:
    def __init__(self) -> None: ...

    @property
    def minimum(self) -> Position: ...

    @minimum.setter
    def minimum(self, arg: Position, /) -> None: ...

    @property
    def maximum(self) -> Position: ...

    @maximum.setter
    def maximum(self, arg: Position, /) -> None: ...

    def get_size(self) -> Position: ...

    def extend(self, arg: Position, /) -> None: ...

    def add_margin(self, arg: float, /) -> None: ...

class FractionalBox:
    def __init__(self) -> None: ...

    @property
    def minimum(self) -> Fractional: ...

    @minimum.setter
    def minimum(self, arg: Fractional, /) -> None: ...

    @property
    def maximum(self) -> Fractional: ...

    @maximum.setter
    def maximum(self, arg: Fractional, /) -> None: ...

    def get_size(self) -> Fractional: ...

    def extend(self, arg: Fractional, /) -> None: ...

    def add_margin(self, arg: float, /) -> None: ...

class FTransform(Transform):
    def __init__(self) -> None: ...

    def apply(self, arg: Fractional, /) -> Fractional: ...  # type: ignore[override]

class NearestImage:
    def dist(self) -> float: ...

    def same_asu(self) -> bool: ...

    def symmetry_code(self, underscore: bool = True) -> str: ...

    @property
    def sym_idx(self) -> int: ...

    @property
    def pbc_shift(self) -> tuple: ...

    def __repr__(self) -> str: ...

class Asu(enum.Enum):
    Same = 0

    Different = 1

    Any = 2

class UnitCell:
    @overload
    def __init__(self) -> None: ...

    @overload
    def __init__(self, a: float, b: float, c: float, alpha: float, beta: float, gamma: float) -> None: ...

    @property
    def a(self) -> float: ...

    @property
    def b(self) -> float: ...

    @property
    def c(self) -> float: ...

    @property
    def alpha(self) -> float: ...

    @property
    def beta(self) -> float: ...

    @property
    def gamma(self) -> float: ...

    @property
    def volume(self) -> float: ...

    @property
    def explicit_matrices(self) -> bool: ...

    @property
    def images(self) -> list[FTransform]: ...

    @property
    def parameters(self) -> tuple: ...

    @property
    def frac(self) -> Transform: ...

    @property
    def orth(self) -> Transform: ...

    def set(self, arg0: float, arg1: float, arg2: float, arg3: float, arg4: float, arg5: float, /) -> None: ...

    def changed_basis_forward(self, op: Op, set_images: bool) -> UnitCell: ...

    def changed_basis_backward(self, op: Op, set_images: bool) -> UnitCell: ...

    def is_compatible_with_spacegroup(self, sg: SpaceGroup, eps: float = 0.001) -> bool: ...

    def is_crystal(self) -> bool: ...

    def approx(self, other: UnitCell, epsilon: float) -> bool: ...

    def is_similar(self, other: UnitCell, rel: float, deg: float) -> bool: ...

    def calculate_u_eq(self, arg: SMat33d, /) -> float: ...

    def fractionalize(self, arg: Position, /) -> Fractional: ...

    def orthogonalize(self, arg: Fractional, /) -> Position: ...

    def orthogonalize_box(self, arg: FractionalBox, /) -> PositionBox: ...

    def op_as_transform(self, arg: Op, /) -> Transform: ...

    def volume_per_image(self) -> float: ...

    def find_nearest_image(self, ref: Position, pos: Position, asu: Asu = Asu.Any) -> NearestImage: ...

    @overload
    def find_nearest_pbc_image(self, fref: Fractional, fpos: Fractional, image_idx: int = 0) -> NearestImage: ...

    @overload
    def find_nearest_pbc_image(self, ref: Position, pos: Position, image_idx: int = 0) -> NearestImage: ...

    def find_nearest_pbc_images(self, fref: Fractional, dist: float, fpos: Fractional, image_idx: int) -> list[NearestImage]: ...

    def find_nearest_pbc_position(self, ref: Position, pos: Position, image_idx: int, inverse: bool = False) -> Position: ...

    def fract_image(self, arg0: NearestImage, arg1: Fractional, /) -> Fractional: ...

    @overload
    def is_special_position(self, pos: Position, max_dist: float = 0.8) -> int: ...

    @overload
    def is_special_position(self, fpos: Fractional, max_dist: float) -> int: ...

    def calculate_1_d2(self, hkl: Sequence[int]) -> float: ...

    def calculate_1_d2_array(self, arg: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), device='cpu')], /) -> NDArray[numpy.float64]: ...

    def calculate_d(self, hkl: Sequence[int]) -> float: ...

    def calculate_d_array(self, arg: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), device='cpu')], /) -> NDArray[numpy.float64]: ...

    def metric_tensor(self) -> SMat33d: ...

    def reciprocal_metric_tensor(self) -> SMat33d: ...

    def reciprocal(self) -> UnitCell: ...

    def get_hkl_limits(self, dmin: float) -> list[int]: ...

    def primitive_orth_matrix(self, centring_type: str) -> Mat33: ...

    def __eq__(self, arg: object, /) -> bool: ...

    def __getstate__(self) -> bytes: ...

    def __setstate__(self, arg: bytes, /) -> None: ...

    def __repr__(self) -> str: ...

class SellingVector:
    @overload
    def __init__(self, arg: Sequence[float], /) -> None: ...

    @overload
    def __init__(self, arg0: UnitCell, arg1: SpaceGroup, /) -> None: ...

    @property
    def parameters(self) -> tuple: ...

    def cell_parameters(self) -> tuple: ...

    def get_cell(self) -> UnitCell: ...

    def sum_b_squared(self) -> float: ...

    def gruber(self) -> GruberVector: ...

    def is_reduced(self, epsilon: float = 1e-09) -> bool: ...

    def reduce_step(self, epsilon: float = 1e-09) -> bool: ...

    def reduce(self, epsilon: float = 1e-09, iteration_limit: int = 100) -> int: ...

    def sort(self, epsilon: float = 1e-09) -> None: ...

    def __repr__(self) -> str: ...

class GruberVector:
    @overload
    def __init__(self, arg: Sequence[float], /) -> None: ...

    @overload
    def __init__(self, cell: UnitCell, sg: SpaceGroup | None, track_change_of_basis: bool = False) -> None: ...

    @overload
    def __init__(self, cell: UnitCell, centring: str, track_change_of_basis: bool = False) -> None: ...

    @property
    def parameters(self) -> tuple: ...

    def cell_parameters(self) -> tuple: ...

    def get_cell(self) -> UnitCell: ...

    @property
    def change_of_basis(self) -> Op: ...

    def selling(self) -> SellingVector: ...

    def is_normalized(self) -> bool: ...

    def is_buerger(self, epsilon: float = 1e-09) -> bool: ...

    def normalize(self, epsilon: float = 1e-09) -> None: ...

    def buerger_reduce(self) -> int: ...

    def niggli_step(self, epsilon: float) -> bool: ...

    def niggli_reduce(self, epsilon: float = 1e-09, iteration_limit: int = 100) -> int: ...

    def is_niggli(self, epsilon: float = 1e-09) -> bool: ...

    def __repr__(self) -> str: ...

def find_lattice_2fold_ops(reduced_cell: UnitCell, max_obliq: float) -> list[tuple[Op, float]]: ...

def find_lattice_symmetry_r(arg0: UnitCell, arg1: float, /) -> GroupOps: ...

def find_lattice_symmetry(cell: UnitCell, centring: str, max_obliq: float) -> GroupOps: ...

def find_twin_laws(cell: UnitCell, sg: SpaceGroup, max_obliq: float, all_ops: bool) -> list[Op]: ...

class IT92Coef:
    @property
    def a(self) -> list[float]: ...

    @property
    def b(self) -> list[float]: ...

    @property
    def c(self) -> float: ...

    def get_coefs(self) -> list[float]: ...

    def set_coefs(self, arg: Sequence[float], /) -> None: ...

    def calculate_sf(self, stol2: float) -> float: ...

    def calculate_density_iso(self, r2: float, B: float) -> float: ...

def IT92_normalize() -> None: ...

def IT92_get_ignore_charge() -> bool: ...

def IT92_set_ignore_charge(arg: bool, /) -> None: ...

class C4322Coef:
    @property
    def a(self) -> list[float]: ...

    @property
    def b(self) -> list[float]: ...

    def get_coefs(self) -> list[float]: ...

    def set_coefs(self, arg: Sequence[float], /) -> None: ...

    def calculate_sf(self, stol2: float) -> float: ...

    def calculate_density_iso(self, r2: float, B: float) -> float: ...

def set_custom_form_factors(arg: Sequence[Sequence[float]], /) -> None: ...

class Neutron92:
    def get_coefs(self) -> list[float]: ...

    def calculate_sf(self, stol2: float) -> float: ...

    def calculate_density_iso(self, r2: float, B: float) -> float: ...

class Element:
    @overload
    def __init__(self, arg: str, /) -> None: ...

    @overload
    def __init__(self, arg: int, /) -> None: ...

    def __eq__(self, arg: object, /) -> bool: ...

    @property
    def name(self) -> str: ...

    @property
    def weight(self) -> float: ...

    @property
    def covalent_r(self) -> float: ...

    @property
    def vdw_r(self) -> float: ...

    @property
    def atomic_number(self) -> int: ...

    @property
    def is_hydrogen(self) -> bool: ...

    @property
    def is_metal(self) -> bool: ...

    @property
    def it92(self) -> IT92Coef: ...

    @property
    def c4322(self) -> C4322Coef: ...

    @property
    def neutron92(self) -> Neutron92: ...

    def __hash__(self) -> int: ...

    def __repr__(self) -> str: ...

def IT92_get_exact(arg0: Element, arg1: int, /) -> IT92Coef: ...

def set_is_metal(arg0: str, arg1: bool, /) -> None: ...

class ResidueKind(enum.Enum):
    UNKNOWN = 0

    AA = 1

    AAD = 2

    PAA = 3

    MAA = 4

    RNA = 5

    DNA = 6

    BUF = 7

    HOH = 8

    PYR = 9

    KET = 10

    ELS = 11

class ResidueInfo:
    @property
    def name(self) -> str: ...

    @property
    def kind(self) -> ResidueKind: ...

    @kind.setter
    def kind(self, arg: ResidueKind, /) -> None: ...

    @property
    def one_letter_code(self) -> str: ...

    @property
    def hydrogen_count(self) -> int: ...

    @property
    def weight(self) -> float: ...

    def found(self) -> bool: ...

    def is_standard(self) -> bool: ...

    def fasta_code(self) -> str: ...

    def is_water(self) -> bool: ...

    def is_nucleic_acid(self) -> bool: ...

    def is_amino_acid(self) -> bool: ...

def find_tabulated_residue(name: str) -> ResidueInfo:
    """Find chemical component information in the internal table."""

def find_tabulated_residue_idx(name: str) -> int: ...

def expand_one_letter(arg0: str, arg1: ResidueKind, /) -> str: ...

def expand_one_letter_sequence(arg0: str, arg1: ResidueKind, /) -> list[str]: ...

def resinfo_table() -> Iterator[ResidueInfo]: ...

class XdsAscii:
    @property
    def source_path(self) -> str: ...

    @property
    def read_columns(self) -> int: ...

    @property
    def spacegroup_number(self) -> int: ...

    @property
    def wavelength(self) -> float: ...

    @property
    def cell_constants(self) -> list[float]: ...

    @property
    def rotation_axis(self) -> Vec3: ...

    @property
    def friedels_law(self) -> str: ...

    @property
    def generated_by(self) -> str: ...

    @property
    def data_size(self) -> int: ...

    @property
    def miller_array(self) -> Annotated[NDArray[numpy.int32], dict(shape=(None, 3))]: ...

    @property
    def iset_array(self) -> Annotated[NDArray[numpy.int32], dict(shape=(None,))]: ...

    @property
    def iobs_array(self) -> Annotated[NDArray[numpy.float64], dict(shape=(None,))]: ...

    @property
    def sigma_array(self) -> Annotated[NDArray[numpy.float64], dict(shape=(None,))]: ...

    @property
    def xd_array(self) -> Annotated[NDArray[numpy.float64], dict(shape=(None,))]: ...

    @property
    def yd_array(self) -> Annotated[NDArray[numpy.float64], dict(shape=(None,))]: ...

    @property
    def zd_array(self) -> Annotated[NDArray[numpy.float64], dict(shape=(None,))]: ...

    def filtered(self, arg: Annotated[NDArray[numpy.bool], dict(shape=(None,), device='cpu')], /) -> XdsAscii: ...

    def apply_polarization_correction(self, p: float, normal: Vec3) -> None: ...

    def to_mtz(self) -> Mtz: ...

def read_xds_ascii(arg: str, /) -> XdsAscii: ...

class SeqId:
    @overload
    def __init__(self, arg0: int, arg1: str, /) -> None: ...

    @overload
    def __init__(self, arg: str, /) -> None: ...

    @property
    def num(self) -> int | None: ...

    @num.setter
    def num(self, arg: int | None | None) -> None: ...

    @property
    def icode(self) -> str: ...

    @icode.setter
    def icode(self, arg: str, /) -> None: ...

    def __str__(self) -> str: ...

    def __repr__(self) -> str: ...

    def __eq__(self, arg: object, /) -> bool: ...

    def __lt__(self, arg: SeqId, /) -> bool: ...

    def __hash__(self) -> int: ...

class ResidueId:
    def __init__(self) -> None: ...

    @property
    def name(self) -> str: ...

    @name.setter
    def name(self, arg: str, /) -> None: ...

    @property
    def seqid(self) -> SeqId: ...

    @seqid.setter
    def seqid(self, arg: SeqId, /) -> None: ...

    @property
    def segment(self) -> str: ...

    @segment.setter
    def segment(self, arg: str, /) -> None: ...

    def __str__(self) -> str: ...

    def __repr__(self) -> str: ...

    def __eq__(self, arg: object, /) -> bool: ...

class AtomAddress:
    @overload
    def __init__(self) -> None: ...

    @overload
    def __init__(self, chain: str, seqid: SeqId, resname: str, atom: str, altloc: str = '\x00') -> None: ...

    @property
    def chain_name(self) -> str: ...

    @chain_name.setter
    def chain_name(self, arg: str, /) -> None: ...

    @property
    def res_id(self) -> ResidueId: ...

    @res_id.setter
    def res_id(self, arg: ResidueId, /) -> None: ...

    @property
    def atom_name(self) -> str: ...

    @atom_name.setter
    def atom_name(self, arg: str, /) -> None: ...

    @property
    def altloc(self) -> str: ...

    @altloc.setter
    def altloc(self, arg: str, /) -> None: ...

    def __str__(self) -> str: ...

    def __repr__(self) -> str: ...

    def __eq__(self, arg: object, /) -> bool: ...

class NcsOp:
    @overload
    def __init__(self) -> None: ...

    @overload
    def __init__(self, tr: Transform, id: str = '', given: bool = False) -> None: ...

    @property
    def id(self) -> str: ...

    @id.setter
    def id(self, arg: str, /) -> None: ...

    @property
    def given(self) -> bool: ...

    @given.setter
    def given(self, arg: bool, /) -> None: ...

    @property
    def tr(self) -> Transform: ...

    def apply(self, arg: Position, /) -> Position: ...

    def __repr__(self) -> str: ...

class NcsOpList:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: NcsOpList) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[NcsOp], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[NcsOp]: ...

    @overload
    def __getitem__(self, arg: int, /) -> NcsOp: ...

    @overload
    def __getitem__(self, arg: slice, /) -> NcsOpList: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: NcsOp, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: NcsOp, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> NcsOp:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: NcsOpList, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: NcsOp, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: NcsOpList, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class EntityType(enum.Enum):
    Unknown = 0

    Polymer = 1

    NonPolymer = 2

    Branched = 3

    Water = 4

class PolymerType(enum.Enum):
    PeptideL = 1

    PeptideD = 2

    Dna = 3

    Rna = 4

    DnaRnaHybrid = 5

    SaccharideD = 6

    SaccharideL = 7

    Pna = 8

    CyclicPseudoPeptide = 9

    Other = 10

    Unknown = 0

class Entity:
    def __init__(self, arg: str, /) -> None: ...

    @property
    def name(self) -> str: ...

    @name.setter
    def name(self, arg: str, /) -> None: ...

    @property
    def subchains(self) -> list[str]: ...

    @subchains.setter
    def subchains(self, arg: Sequence[str], /) -> None: ...

    @property
    def entity_type(self) -> EntityType: ...

    @entity_type.setter
    def entity_type(self, arg: EntityType, /) -> None: ...

    @property
    def polymer_type(self) -> PolymerType: ...

    @polymer_type.setter
    def polymer_type(self, arg: PolymerType, /) -> None: ...

    @property
    def sifts_unp_acc(self) -> list[str]: ...

    @sifts_unp_acc.setter
    def sifts_unp_acc(self, arg: Sequence[str], /) -> None: ...

    @property
    def full_sequence(self) -> list[str]: ...

    @full_sequence.setter
    def full_sequence(self, arg: Sequence[str], /) -> None: ...

    @staticmethod
    def first_mon(arg: str, /) -> str: ...

    def __repr__(self) -> str: ...

class EntityList:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: EntityList) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Entity], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Entity]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Entity: ...

    @overload
    def __getitem__(self, arg: slice, /) -> EntityList: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Entity, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Entity, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Entity:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: EntityList, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Entity, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: EntityList, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class ConnectionType(enum.Enum):
    Covale = 0

    Disulf = 1

    Hydrog = 2

    MetalC = 3

    Unknown = 4

class Connection:
    def __init__(self) -> None: ...

    @property
    def name(self) -> str: ...

    @name.setter
    def name(self, arg: str, /) -> None: ...

    @property
    def link_id(self) -> str: ...

    @link_id.setter
    def link_id(self, arg: str, /) -> None: ...

    @property
    def type(self) -> ConnectionType: ...

    @type.setter
    def type(self, arg: ConnectionType, /) -> None: ...

    @property
    def asu(self) -> Asu: ...

    @asu.setter
    def asu(self, arg: Asu, /) -> None: ...

    @property
    def partner1(self) -> AtomAddress: ...

    @partner1.setter
    def partner1(self, arg: AtomAddress, /) -> None: ...

    @property
    def partner2(self) -> AtomAddress: ...

    @partner2.setter
    def partner2(self, arg: AtomAddress, /) -> None: ...

    @property
    def reported_distance(self) -> float: ...

    @reported_distance.setter
    def reported_distance(self, arg: float, /) -> None: ...

    def __repr__(self) -> str: ...

class ConnectionList:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: ConnectionList) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Connection], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Connection]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Connection: ...

    @overload
    def __getitem__(self, arg: slice, /) -> ConnectionList: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Connection, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Connection, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Connection:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: ConnectionList, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Connection, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: ConnectionList, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class CisPep:
    def __init__(self) -> None: ...

    @property
    def partner_c(self) -> AtomAddress: ...

    @partner_c.setter
    def partner_c(self, arg: AtomAddress, /) -> None: ...

    @property
    def partner_n(self) -> AtomAddress: ...

    @partner_n.setter
    def partner_n(self, arg: AtomAddress, /) -> None: ...

    @property
    def model_num(self) -> int: ...

    @model_num.setter
    def model_num(self, arg: int, /) -> None: ...

    @property
    def only_altloc(self) -> str: ...

    @only_altloc.setter
    def only_altloc(self, arg: str, /) -> None: ...

    @property
    def reported_angle(self) -> float: ...

    @reported_angle.setter
    def reported_angle(self, arg: float, /) -> None: ...

class ModRes:
    def __init__(self) -> None: ...

    @property
    def chain_name(self) -> str: ...

    @chain_name.setter
    def chain_name(self, arg: str, /) -> None: ...

    @property
    def res_id(self) -> ResidueId: ...

    @res_id.setter
    def res_id(self, arg: ResidueId, /) -> None: ...

    @property
    def parent_comp_id(self) -> str: ...

    @parent_comp_id.setter
    def parent_comp_id(self, arg: str, /) -> None: ...

    @property
    def mod_id(self) -> str: ...

    @mod_id.setter
    def mod_id(self, arg: str, /) -> None: ...

    @property
    def details(self) -> str: ...

    @details.setter
    def details(self, arg: str, /) -> None: ...

class Helix:
    def __init__(self) -> None: ...

    class HelixClass(enum.Enum):
        UnknownHelix = 0

        RAlpha = 1

        ROmega = 2

        RPi = 3

        RGamma = 4

        R310 = 5

        LAlpha = 6

        LOmega = 7

        LGamma = 8

        Helix27 = 9

        HelixPolyProlineNone = 10

    @property
    def start(self) -> AtomAddress: ...

    @start.setter
    def start(self, arg: AtomAddress, /) -> None: ...

    @property
    def end(self) -> AtomAddress: ...

    @end.setter
    def end(self, arg: AtomAddress, /) -> None: ...

    @property
    def pdb_helix_class(self) -> Helix.HelixClass: ...

    @pdb_helix_class.setter
    def pdb_helix_class(self, arg: Helix.HelixClass, /) -> None: ...

    @property
    def length(self) -> int: ...

    @length.setter
    def length(self, arg: int, /) -> None: ...

class HelixList:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: HelixList) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Helix], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Helix]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Helix: ...

    @overload
    def __getitem__(self, arg: slice, /) -> HelixList: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Helix, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Helix, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Helix:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: HelixList, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Helix, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: HelixList, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class Sheet:
    def __init__(self, arg: str, /) -> None: ...

    class Strand:
        def __init__(self) -> None: ...

        @property
        def start(self) -> AtomAddress: ...

        @start.setter
        def start(self, arg: AtomAddress, /) -> None: ...

        @property
        def end(self) -> AtomAddress: ...

        @end.setter
        def end(self, arg: AtomAddress, /) -> None: ...

        @property
        def hbond_atom2(self) -> AtomAddress: ...

        @hbond_atom2.setter
        def hbond_atom2(self, arg: AtomAddress, /) -> None: ...

        @property
        def hbond_atom1(self) -> AtomAddress: ...

        @hbond_atom1.setter
        def hbond_atom1(self, arg: AtomAddress, /) -> None: ...

        @property
        def sense(self) -> int: ...

        @sense.setter
        def sense(self, arg: int, /) -> None: ...

        @property
        def name(self) -> str: ...

        @name.setter
        def name(self, arg: str, /) -> None: ...

    class StrandList:
        @overload
        def __init__(self) -> None:
            """Default constructor"""

        @overload
        def __init__(self, arg: Sheet.StrandList) -> None:
            """Copy constructor"""

        @overload
        def __init__(self, arg: Iterable[Sheet.Strand], /) -> None:
            """Construct from an iterable object"""

        def __len__(self) -> int: ...

        def __bool__(self) -> bool:
            """Check whether the vector is nonempty"""

        def __repr__(self) -> str: ...

        def __iter__(self) -> Iterator[Sheet.Strand]: ...

        @overload
        def __getitem__(self, arg: int, /) -> Sheet.Strand: ...

        @overload
        def __getitem__(self, arg: slice, /) -> Sheet.StrandList: ...

        def clear(self) -> None:
            """Remove all items from list."""

        def append(self, arg: Sheet.Strand, /) -> None:
            """Append `arg` to the end of the list."""

        def insert(self, arg0: int, arg1: Sheet.Strand, /) -> None:
            """Insert object `arg1` before index `arg0`."""

        def pop(self, index: int = -1) -> Sheet.Strand:
            """Remove and return item at `index` (default last)."""

        def extend(self, arg: Sheet.StrandList, /) -> None:
            """Extend `self` by appending elements from `arg`."""

        @overload
        def __setitem__(self, arg0: int, arg1: Sheet.Strand, /) -> None: ...

        @overload
        def __setitem__(self, arg0: slice, arg1: Sheet.StrandList, /) -> None: ...

        @overload
        def __delitem__(self, arg: int, /) -> None: ...

        @overload
        def __delitem__(self, arg: slice, /) -> None: ...

    @property
    def name(self) -> str: ...

    @name.setter
    def name(self, arg: str, /) -> None: ...

    @property
    def strands(self) -> Sheet.StrandList: ...

    @strands.setter
    def strands(self, arg: Sheet.StrandList, /) -> None: ...

class SheetList:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: SheetList) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Sheet], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Sheet]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Sheet: ...

    @overload
    def __getitem__(self, arg: slice, /) -> SheetList: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Sheet, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Sheet, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Sheet:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: SheetList, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Sheet, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: SheetList, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class Assembly:
    def __init__(self, arg: str, /) -> None: ...

    class Operator:
        def __init__(self) -> None: ...

        @property
        def name(self) -> str: ...

        @name.setter
        def name(self, arg: str, /) -> None: ...

        @property
        def type(self) -> str: ...

        @type.setter
        def type(self, arg: str, /) -> None: ...

        @property
        def transform(self) -> Transform: ...

        @transform.setter
        def transform(self, arg: Transform, /) -> None: ...

    class OperatorList:
        @overload
        def __init__(self) -> None:
            """Default constructor"""

        @overload
        def __init__(self, arg: Assembly.OperatorList) -> None:
            """Copy constructor"""

        @overload
        def __init__(self, arg: Iterable[Assembly.Operator], /) -> None:
            """Construct from an iterable object"""

        def __len__(self) -> int: ...

        def __bool__(self) -> bool:
            """Check whether the vector is nonempty"""

        def __repr__(self) -> str: ...

        def __iter__(self) -> Iterator[Assembly.Operator]: ...

        @overload
        def __getitem__(self, arg: int, /) -> Assembly.Operator: ...

        @overload
        def __getitem__(self, arg: slice, /) -> Assembly.OperatorList: ...

        def clear(self) -> None:
            """Remove all items from list."""

        def append(self, arg: Assembly.Operator, /) -> None:
            """Append `arg` to the end of the list."""

        def insert(self, arg0: int, arg1: Assembly.Operator, /) -> None:
            """Insert object `arg1` before index `arg0`."""

        def pop(self, index: int = -1) -> Assembly.Operator:
            """Remove and return item at `index` (default last)."""

        def extend(self, arg: Assembly.OperatorList, /) -> None:
            """Extend `self` by appending elements from `arg`."""

        @overload
        def __setitem__(self, arg0: int, arg1: Assembly.Operator, /) -> None: ...

        @overload
        def __setitem__(self, arg0: slice, arg1: Assembly.OperatorList, /) -> None: ...

        @overload
        def __delitem__(self, arg: int, /) -> None: ...

        @overload
        def __delitem__(self, arg: slice, /) -> None: ...

    class Gen:
        def __init__(self) -> None: ...

        @property
        def chains(self) -> list[str]: ...

        @chains.setter
        def chains(self, arg: Sequence[str], /) -> None: ...

        @property
        def subchains(self) -> list[str]: ...

        @subchains.setter
        def subchains(self, arg: Sequence[str], /) -> None: ...

        @property
        def operators(self) -> Assembly.OperatorList: ...

    class GenList:
        @overload
        def __init__(self) -> None:
            """Default constructor"""

        @overload
        def __init__(self, arg: Assembly.GenList) -> None:
            """Copy constructor"""

        @overload
        def __init__(self, arg: Iterable[Assembly.Gen], /) -> None:
            """Construct from an iterable object"""

        def __len__(self) -> int: ...

        def __bool__(self) -> bool:
            """Check whether the vector is nonempty"""

        def __repr__(self) -> str: ...

        def __iter__(self) -> Iterator[Assembly.Gen]: ...

        @overload
        def __getitem__(self, arg: int, /) -> Assembly.Gen: ...

        @overload
        def __getitem__(self, arg: slice, /) -> Assembly.GenList: ...

        def clear(self) -> None:
            """Remove all items from list."""

        def append(self, arg: Assembly.Gen, /) -> None:
            """Append `arg` to the end of the list."""

        def insert(self, arg0: int, arg1: Assembly.Gen, /) -> None:
            """Insert object `arg1` before index `arg0`."""

        def pop(self, index: int = -1) -> Assembly.Gen:
            """Remove and return item at `index` (default last)."""

        def extend(self, arg: Assembly.GenList, /) -> None:
            """Extend `self` by appending elements from `arg`."""

        @overload
        def __setitem__(self, arg0: int, arg1: Assembly.Gen, /) -> None: ...

        @overload
        def __setitem__(self, arg0: slice, arg1: Assembly.GenList, /) -> None: ...

        @overload
        def __delitem__(self, arg: int, /) -> None: ...

        @overload
        def __delitem__(self, arg: slice, /) -> None: ...

    @property
    def name(self) -> str: ...

    @name.setter
    def name(self, arg: str, /) -> None: ...

    @property
    def author_determined(self) -> bool: ...

    @author_determined.setter
    def author_determined(self, arg: bool, /) -> None: ...

    @property
    def software_determined(self) -> bool: ...

    @software_determined.setter
    def software_determined(self, arg: bool, /) -> None: ...

    @property
    def oligomeric_details(self) -> str: ...

    @oligomeric_details.setter
    def oligomeric_details(self, arg: str, /) -> None: ...

    @property
    def generators(self) -> Assembly.GenList: ...

    @property
    def special_kind(self) -> AssemblySpecialKind: ...

    @special_kind.setter
    def special_kind(self, arg: AssemblySpecialKind, /) -> None: ...

    def __repr__(self) -> str: ...

class AssemblySpecialKind(enum.Enum):
    NA = 0

    CompleteIcosahedral = 1

    RepresentativeHelical = 2

    CompletePoint = 3

class AssemblyList:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: AssemblyList) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Assembly], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Assembly]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Assembly: ...

    @overload
    def __getitem__(self, arg: slice, /) -> AssemblyList: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Assembly, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Assembly, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Assembly:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: AssemblyList, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Assembly, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: AssemblyList, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class SoftwareItem:
    def __init__(self) -> None: ...

    class Classification(enum.Enum):
        DataCollection = 0

        DataExtraction = 1

        DataProcessing = 2

        DataReduction = 3

        DataScaling = 4

        ModelBuilding = 5

        Phasing = 6

        Refinement = 7

        Unspecified = 8

    @property
    def name(self) -> str: ...

    @name.setter
    def name(self, arg: str, /) -> None: ...

    @property
    def version(self) -> str: ...

    @version.setter
    def version(self, arg: str, /) -> None: ...

    @property
    def date(self) -> str: ...

    @date.setter
    def date(self, arg: str, /) -> None: ...

    @property
    def classification(self) -> SoftwareItem.Classification: ...

    @classification.setter
    def classification(self, arg: SoftwareItem.Classification, /) -> None: ...

    @property
    def contact_author(self) -> str: ...

    @contact_author.setter
    def contact_author(self, arg: str, /) -> None: ...

    @property
    def contact_author_email(self) -> str: ...

    @contact_author_email.setter
    def contact_author_email(self, arg: str, /) -> None: ...

class ReflectionsInfo:
    def __init__(self) -> None: ...

    @property
    def resolution_high(self) -> float: ...

    @resolution_high.setter
    def resolution_high(self, arg: float, /) -> None: ...

    @property
    def resolution_low(self) -> float: ...

    @resolution_low.setter
    def resolution_low(self, arg: float, /) -> None: ...

    @property
    def completeness(self) -> float: ...

    @completeness.setter
    def completeness(self, arg: float, /) -> None: ...

    @property
    def redundancy(self) -> float: ...

    @redundancy.setter
    def redundancy(self, arg: float, /) -> None: ...

    @property
    def r_merge(self) -> float: ...

    @r_merge.setter
    def r_merge(self, arg: float, /) -> None: ...

    @property
    def r_sym(self) -> float: ...

    @r_sym.setter
    def r_sym(self, arg: float, /) -> None: ...

    @property
    def mean_I_over_sigma(self) -> float: ...

    @mean_I_over_sigma.setter
    def mean_I_over_sigma(self, arg: float, /) -> None: ...

class ExperimentInfo:
    def __init__(self) -> None: ...

    @property
    def method(self) -> str: ...

    @method.setter
    def method(self, arg: str, /) -> None: ...

    @property
    def number_of_crystals(self) -> int: ...

    @number_of_crystals.setter
    def number_of_crystals(self, arg: int, /) -> None: ...

    @property
    def unique_reflections(self) -> int: ...

    @unique_reflections.setter
    def unique_reflections(self, arg: int, /) -> None: ...

    @property
    def reflections(self) -> ReflectionsInfo: ...

    @reflections.setter
    def reflections(self, arg: ReflectionsInfo, /) -> None: ...

    @property
    def b_wilson(self) -> float: ...

    @b_wilson.setter
    def b_wilson(self, arg: float, /) -> None: ...

    @property
    def shells(self) -> list[ReflectionsInfo]: ...

    @shells.setter
    def shells(self, arg: Sequence[ReflectionsInfo], /) -> None: ...

    @property
    def diffraction_ids(self) -> list[str]: ...

    @diffraction_ids.setter
    def diffraction_ids(self, arg: Sequence[str], /) -> None: ...

class DiffractionInfo:
    def __init__(self) -> None: ...

    @property
    def id(self) -> str: ...

    @id.setter
    def id(self, arg: str, /) -> None: ...

    @property
    def temperature(self) -> float: ...

    @temperature.setter
    def temperature(self, arg: float, /) -> None: ...

    @property
    def source(self) -> str: ...

    @source.setter
    def source(self, arg: str, /) -> None: ...

    @property
    def source_type(self) -> str: ...

    @source_type.setter
    def source_type(self, arg: str, /) -> None: ...

    @property
    def synchrotron(self) -> str: ...

    @synchrotron.setter
    def synchrotron(self, arg: str, /) -> None: ...

    @property
    def beamline(self) -> str: ...

    @beamline.setter
    def beamline(self, arg: str, /) -> None: ...

    @property
    def wavelengths(self) -> str: ...

    @wavelengths.setter
    def wavelengths(self, arg: str, /) -> None: ...

    @property
    def scattering_type(self) -> str: ...

    @scattering_type.setter
    def scattering_type(self, arg: str, /) -> None: ...

    @property
    def mono_or_laue(self) -> str: ...

    @mono_or_laue.setter
    def mono_or_laue(self, arg: str, /) -> None: ...

    @property
    def monochromator(self) -> str: ...

    @monochromator.setter
    def monochromator(self, arg: str, /) -> None: ...

    @property
    def collection_date(self) -> str: ...

    @collection_date.setter
    def collection_date(self, arg: str, /) -> None: ...

    @property
    def optics(self) -> str: ...

    @optics.setter
    def optics(self, arg: str, /) -> None: ...

    @property
    def detector(self) -> str: ...

    @detector.setter
    def detector(self, arg: str, /) -> None: ...

    @property
    def detector_make(self) -> str: ...

    @detector_make.setter
    def detector_make(self, arg: str, /) -> None: ...

class CrystalInfo:
    def __init__(self) -> None: ...

    @property
    def id(self) -> str: ...

    @id.setter
    def id(self, arg: str, /) -> None: ...

    @property
    def description(self) -> str: ...

    @description.setter
    def description(self, arg: str, /) -> None: ...

    @property
    def ph(self) -> float: ...

    @ph.setter
    def ph(self, arg: float, /) -> None: ...

    @property
    def ph_range(self) -> str: ...

    @ph_range.setter
    def ph_range(self, arg: str, /) -> None: ...

    @property
    def diffractions(self) -> list[DiffractionInfo]: ...

    @diffractions.setter
    def diffractions(self, arg: Sequence[DiffractionInfo], /) -> None: ...

class TlsGroup:
    def __init__(self) -> None: ...

    class Selection:
        def __init__(self) -> None: ...

        @property
        def chain(self) -> str: ...

        @chain.setter
        def chain(self, arg: str, /) -> None: ...

        @property
        def res_begin(self) -> SeqId: ...

        @res_begin.setter
        def res_begin(self, arg: SeqId, /) -> None: ...

        @property
        def res_end(self) -> SeqId: ...

        @res_end.setter
        def res_end(self, arg: SeqId, /) -> None: ...

        @property
        def details(self) -> str: ...

        @details.setter
        def details(self, arg: str, /) -> None: ...

    @property
    def id(self) -> str: ...

    @id.setter
    def id(self, arg: str, /) -> None: ...

    @property
    def selections(self) -> list[TlsGroup.Selection]: ...

    @selections.setter
    def selections(self, arg: Sequence[TlsGroup.Selection], /) -> None: ...

    @property
    def origin(self) -> Position: ...

    @origin.setter
    def origin(self, arg: Position, /) -> None: ...

    @property
    def T(self) -> SMat33d: ...

    @T.setter
    def T(self, arg: SMat33d, /) -> None: ...

    @property
    def L(self) -> SMat33d: ...

    @L.setter
    def L(self, arg: SMat33d, /) -> None: ...

    @property
    def S(self) -> Mat33: ...

    @S.setter
    def S(self, arg: Mat33, /) -> None: ...

class BasicRefinementInfo:
    def __init__(self) -> None: ...

    @property
    def resolution_high(self) -> float: ...

    @resolution_high.setter
    def resolution_high(self, arg: float, /) -> None: ...

    @property
    def resolution_low(self) -> float: ...

    @resolution_low.setter
    def resolution_low(self, arg: float, /) -> None: ...

    @property
    def completeness(self) -> float: ...

    @completeness.setter
    def completeness(self, arg: float, /) -> None: ...

    @property
    def reflection_count(self) -> int: ...

    @reflection_count.setter
    def reflection_count(self, arg: int, /) -> None: ...

    @property
    def work_set_count(self) -> int: ...

    @work_set_count.setter
    def work_set_count(self, arg: int, /) -> None: ...

    @property
    def rfree_set_count(self) -> int: ...

    @rfree_set_count.setter
    def rfree_set_count(self, arg: int, /) -> None: ...

    @property
    def r_all(self) -> float: ...

    @r_all.setter
    def r_all(self, arg: float, /) -> None: ...

    @property
    def r_work(self) -> float: ...

    @r_work.setter
    def r_work(self, arg: float, /) -> None: ...

    @property
    def r_free(self) -> float: ...

    @r_free.setter
    def r_free(self, arg: float, /) -> None: ...

    @property
    def cc_fo_fc_work(self) -> float: ...

    @cc_fo_fc_work.setter
    def cc_fo_fc_work(self, arg: float, /) -> None: ...

    @property
    def cc_fo_fc_free(self) -> float: ...

    @cc_fo_fc_free.setter
    def cc_fo_fc_free(self, arg: float, /) -> None: ...

    @property
    def fsc_work(self) -> float: ...

    @fsc_work.setter
    def fsc_work(self, arg: float, /) -> None: ...

    @property
    def fsc_free(self) -> float: ...

    @fsc_free.setter
    def fsc_free(self, arg: float, /) -> None: ...

    @property
    def cc_intensity_work(self) -> float: ...

    @cc_intensity_work.setter
    def cc_intensity_work(self, arg: float, /) -> None: ...

    @property
    def cc_intensity_free(self) -> float: ...

    @cc_intensity_free.setter
    def cc_intensity_free(self, arg: float, /) -> None: ...

class RefinementInfo(BasicRefinementInfo):
    def __init__(self) -> None: ...

    class Restr:
        def __init__(self, arg: str, /) -> None: ...

        @property
        def name(self) -> str: ...

        @name.setter
        def name(self, arg: str, /) -> None: ...

        @property
        def count(self) -> int: ...

        @count.setter
        def count(self, arg: int, /) -> None: ...

        @property
        def weight(self) -> float: ...

        @weight.setter
        def weight(self, arg: float, /) -> None: ...

        @property
        def function(self) -> str: ...

        @function.setter
        def function(self, arg: str, /) -> None: ...

        @property
        def dev_ideal(self) -> float: ...

        @dev_ideal.setter
        def dev_ideal(self, arg: float, /) -> None: ...

    @property
    def id(self) -> str: ...

    @id.setter
    def id(self, arg: str, /) -> None: ...

    @property
    def cross_validation_method(self) -> str: ...

    @cross_validation_method.setter
    def cross_validation_method(self, arg: str, /) -> None: ...

    @property
    def rfree_selection_method(self) -> str: ...

    @rfree_selection_method.setter
    def rfree_selection_method(self, arg: str, /) -> None: ...

    @property
    def bin_count(self) -> int: ...

    @bin_count.setter
    def bin_count(self, arg: int, /) -> None: ...

    @property
    def bins(self) -> list[BasicRefinementInfo]: ...

    @bins.setter
    def bins(self, arg: Sequence[BasicRefinementInfo], /) -> None: ...

    @property
    def mean_b(self) -> float: ...

    @mean_b.setter
    def mean_b(self, arg: float, /) -> None: ...

    @property
    def aniso_b(self) -> SMat33d: ...

    @aniso_b.setter
    def aniso_b(self, arg: SMat33d, /) -> None: ...

    @property
    def luzzati_error(self) -> float: ...

    @luzzati_error.setter
    def luzzati_error(self, arg: float, /) -> None: ...

    @property
    def dpi_blow_r(self) -> float: ...

    @dpi_blow_r.setter
    def dpi_blow_r(self, arg: float, /) -> None: ...

    @property
    def dpi_blow_rfree(self) -> float: ...

    @dpi_blow_rfree.setter
    def dpi_blow_rfree(self, arg: float, /) -> None: ...

    @property
    def dpi_cruickshank_r(self) -> float: ...

    @dpi_cruickshank_r.setter
    def dpi_cruickshank_r(self, arg: float, /) -> None: ...

    @property
    def dpi_cruickshank_rfree(self) -> float: ...

    @dpi_cruickshank_rfree.setter
    def dpi_cruickshank_rfree(self, arg: float, /) -> None: ...

    @property
    def restr_stats(self) -> list[RefinementInfo.Restr]: ...

    @restr_stats.setter
    def restr_stats(self, arg: Sequence[RefinementInfo.Restr], /) -> None: ...

    @property
    def tls_groups(self) -> list[TlsGroup]: ...

    @tls_groups.setter
    def tls_groups(self, arg: Sequence[TlsGroup], /) -> None: ...

    @property
    def remarks(self) -> str: ...

    @remarks.setter
    def remarks(self, arg: str, /) -> None: ...

class Metadata:
    @property
    def authors(self) -> list[str]: ...

    @authors.setter
    def authors(self, arg: Sequence[str], /) -> None: ...

    @property
    def experiments(self) -> list[ExperimentInfo]: ...

    @experiments.setter
    def experiments(self, arg: Sequence[ExperimentInfo], /) -> None: ...

    @property
    def crystals(self) -> list[CrystalInfo]: ...

    @crystals.setter
    def crystals(self, arg: Sequence[CrystalInfo], /) -> None: ...

    @property
    def refinement(self) -> list[RefinementInfo]: ...

    @refinement.setter
    def refinement(self, arg: Sequence[RefinementInfo], /) -> None: ...

    @property
    def software(self) -> list[SoftwareItem]: ...

    @software.setter
    def software(self, arg: Sequence[SoftwareItem], /) -> None: ...

    @property
    def solved_by(self) -> str: ...

    @solved_by.setter
    def solved_by(self, arg: str, /) -> None: ...

    @property
    def starting_model(self) -> str: ...

    @starting_model.setter
    def starting_model(self, arg: str, /) -> None: ...

    @property
    def remark_300_detail(self) -> str: ...

    @remark_300_detail.setter
    def remark_300_detail(self, arg: str, /) -> None: ...

class Atom:
    def __init__(self) -> None: ...

    @property
    def name(self) -> str: ...

    @name.setter
    def name(self, arg: str, /) -> None: ...

    @property
    def altloc(self) -> str: ...

    @altloc.setter
    def altloc(self, arg: str, /) -> None: ...

    @property
    def charge(self) -> int: ...

    @charge.setter
    def charge(self, arg: int, /) -> None: ...

    @property
    def element(self) -> Element: ...

    @element.setter
    def element(self, arg: Element, /) -> None: ...

    @property
    def pos(self) -> Position: ...

    @pos.setter
    def pos(self, arg: Position | Sequence[float], /) -> None: ...

    @property
    def occ(self) -> float: ...

    @occ.setter
    def occ(self, arg: float, /) -> None: ...

    @property
    def b_iso(self) -> float: ...

    @b_iso.setter
    def b_iso(self, arg: float, /) -> None: ...

    @property
    def aniso(self) -> SMat33f: ...

    @aniso.setter
    def aniso(self, arg: SMat33f, /) -> None: ...

    @property
    def serial(self) -> int: ...

    @serial.setter
    def serial(self, arg: int, /) -> None: ...

    @property
    def fraction(self) -> float: ...

    @fraction.setter
    def fraction(self, arg: float, /) -> None: ...

    @property
    def calc_flag(self) -> CalcFlag: ...

    @calc_flag.setter
    def calc_flag(self, arg: CalcFlag, /) -> None: ...

    @property
    def flag(self) -> str: ...

    @flag.setter
    def flag(self, arg: str, /) -> None: ...

    @property
    def tls_group_id(self) -> int: ...

    @tls_group_id.setter
    def tls_group_id(self, arg: int, /) -> None: ...

    def is_hydrogen(self) -> bool: ...

    def has_altloc(self) -> bool: ...

    def b_eq(self) -> float: ...

    def padded_name(self) -> str: ...

    def clone(self) -> Atom: ...

    def __getstate__(self) -> bytes: ...

    def __setstate__(self, arg: bytes, /) -> None: ...

    def __repr__(self) -> str: ...

class Residue(ResidueId):
    def __init__(self) -> None: ...

    @property
    def subchain(self) -> str: ...

    @subchain.setter
    def subchain(self, arg: str, /) -> None: ...

    @property
    def entity_id(self) -> str: ...

    @entity_id.setter
    def entity_id(self, arg: str, /) -> None: ...

    @property
    def label_seq(self) -> int | None: ...

    @label_seq.setter
    def label_seq(self, arg: int | None | None) -> None: ...

    @property
    def entity_type(self) -> EntityType: ...

    @entity_type.setter
    def entity_type(self, arg: EntityType, /) -> None: ...

    @property
    def het_flag(self) -> str: ...

    @het_flag.setter
    def het_flag(self, arg: str, /) -> None: ...

    @property
    def flag(self) -> str: ...

    @flag.setter
    def flag(self, arg: str, /) -> None: ...

    @property
    def sifts_unp(self) -> tuple: ...

    def __len__(self) -> int: ...

    def __contains__(self, arg: str, /) -> bool: ...

    def __iter__(self) -> Iterator[Atom]: ...

    @overload
    def __getitem__(self, index: int) -> Atom: ...

    @overload
    def __getitem__(self, name: str) -> AtomGroup: ...

    @overload
    def __delitem__(self, index: int) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

    def find_atom(self, name: str, altloc: str, el: Element = ...) -> Atom: ...

    def remove_atom(self, name: str, altloc: str, el: Element = ...) -> None: ...

    def add_atom(self, atom: Atom, pos: int = -1) -> Atom: ...

    def first_conformer(self) -> FirstConformerAtoms: ...

    def sole_atom(self, arg: str, /) -> Atom: ...

    def get_ca(self) -> Atom: ...

    def get_p(self) -> Atom: ...

    def is_water(self) -> bool: ...

    def remove_hydrogens(self) -> None: ...

    def trim_to_alanine(self) -> bool: ...

    def recommended_het_flag(self) -> str: ...

    def clone(self) -> Residue: ...

    def __getstate__(self) -> bytes: ...

    def __setstate__(self, arg: bytes, /) -> None: ...

    def __repr__(self) -> str: ...

class Chain:
    def __init__(self, arg: str, /) -> None: ...

    @property
    def name(self) -> str: ...

    @name.setter
    def name(self, arg: str, /) -> None: ...

    def __len__(self) -> int: ...

    def __iter__(self) -> Iterator[Residue]: ...

    @overload
    def __getitem__(self, pdb_seqid: str) -> ResidueGroup: ...

    @overload
    def __getitem__(self, index: int) -> Residue: ...

    @overload
    def __getitem__(self, arg: slice, /) -> list: ...

    @overload
    def __delitem__(self, index: int) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

    def add_residue(self, residue: Residue, pos: int = -1) -> Residue: ...

    def subchains(self) -> list[ResidueSpan]: ...

    def whole(self) -> ResidueSpan: ...

    def get_polymer(self) -> ResidueSpan: ...

    def get_ligands(self) -> ResidueSpan: ...

    def get_waters(self) -> ResidueSpan: ...

    def get_subchain(self, arg: str, /) -> ResidueSpan: ...

    def previous_residue(self, arg: Residue, /) -> Residue: ...

    def next_residue(self, arg: Residue, /) -> Residue: ...

    def append_residues(self, new_residues: Sequence[Residue], min_sep: int = 0) -> None: ...

    def count_atom_sites(self, sel: Selection | None = None) -> int: ...

    def count_occupancies(self, sel: Selection | None = None) -> float: ...

    def calculate_mass(self) -> float: ...

    def calculate_center_of_mass(self) -> Position: ...

    def trim_to_alanine(self) -> None: ...

    def first_conformer(self) -> FirstConformerRes: ...

    def clone(self) -> Chain: ...

    def __getstate__(self) -> bytes: ...

    def __setstate__(self, arg: bytes, /) -> None: ...

    def __repr__(self) -> str: ...

class Model:
    def __init__(self, arg: int, /) -> None: ...

    @property
    def num(self) -> int: ...

    @num.setter
    def num(self, arg: int, /) -> None: ...

    def __len__(self) -> int: ...

    def __iter__(self) -> Iterator[Chain]: ...

    @overload
    def __getitem__(self, index: int) -> Chain: ...

    @overload
    def __getitem__(self, name: str) -> Chain: ...

    def all(self) -> CraGenerator: ...

    def get_subchain(self, name: str) -> ResidueSpan: ...

    def subchains(self) -> list[ResidueSpan]: ...

    def find_residue_group(self, chain: str, seqid: SeqId) -> ResidueGroup: ...

    def sole_residue(self, chain: str, seqid: SeqId) -> Residue: ...

    def get_all_residue_names(self) -> list[str]: ...

    def find_cra(self, arg0: AtomAddress, ignore_segment: bool = False) -> CRA: ...

    def find_chain(self, name: str) -> Chain: ...

    def find_last_chain(self, name: str) -> Chain: ...

    @overload
    def add_chain(self, chain: Chain, pos: int = -1, unique_name: bool = False) -> Chain: ...

    @overload
    def add_chain(self, name: str) -> Chain: ...

    def remove_chain(self, name: str) -> None: ...

    @overload
    def __delitem__(self, name: str) -> None: ...

    @overload
    def __delitem__(self, index: int) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

    def remove_alternative_conformations(self) -> None: ...

    def remove_hydrogens(self) -> None: ...

    def remove_waters(self) -> None: ...

    def remove_ligands_and_waters(self) -> None: ...

    def has_hydrogen(self) -> bool: ...

    def count_atom_sites(self, sel: Selection | None = None) -> int: ...

    def count_occupancies(self, sel: Selection | None = None) -> float: ...

    def calculate_mass(self) -> float: ...

    def calculate_center_of_mass(self) -> Position: ...

    def calculate_b_iso_range(self) -> tuple[float, float]: ...

    def calculate_b_aniso_range(self) -> tuple[float, float]: ...

    def transform_pos_and_adp(self, tr: Transform) -> None: ...

    def split_chains_by_segments(self, arg: HowToNameCopiedChain, /) -> None: ...

    def clone(self) -> Model: ...

    def __getstate__(self) -> bytes: ...

    def __setstate__(self, arg: bytes, /) -> None: ...

    def __repr__(self) -> str: ...

class ResidueSpan:
    def __len__(self) -> int: ...

    def __iter__(self) -> Iterator[Residue]: ...

    def __bool__(self) -> bool: ...

    @overload
    def __getitem__(self, index: int) -> Residue: ...

    @overload
    def __getitem__(self, pdb_seqid: str) -> ResidueGroup: ...

    def __delitem__(self, index: int) -> None: ...

    def add_residue(self, residue: Residue, pos: int = -1) -> Residue: ...

    def first_conformer(self) -> FirstConformerResSpan: ...

    def residue_groups(self) -> ResidueSpanGroups: ...

    def length(self) -> int: ...

    def subchain_id(self) -> str: ...

    def extract_sequence(self) -> list[str]: ...

    def label_seq_id_to_auth(self, arg: int | None, /) -> SeqId: ...

    def auth_seq_id_to_label(self, arg: SeqId, /) -> int | None: ...

    def check_polymer_type(self) -> PolymerType: ...

    def make_one_letter_sequence(self) -> str: ...

    def transform_pos_and_adp(self, arg: Transform, /) -> None: ...

    def __repr__(self) -> str: ...

class ResidueGroup(ResidueSpan):
    @overload  # type: ignore[override]
    def __getitem__(self, index: int) -> Residue: ...

    @overload
    def __getitem__(self, name: str) -> Residue: ...

    def __delitem__(self, name: str) -> None: ...  # type: ignore[override]

    def __repr__(self) -> str: ...

class Selection:
    @overload
    def __init__(self) -> None: ...

    @overload
    def __init__(self, arg: str, /) -> None: ...

    def models(self, arg: Structure, /) -> SelectionModelsProxy: ...

    def chains(self, arg: Model, /) -> SelectionChainsProxy: ...

    def residues(self, arg: Chain, /) -> SelectionResiduesProxy: ...

    def atoms(self, arg: Residue, /) -> SelectionAtomsProxy: ...

    def first_in_model(self, arg: Model, /) -> CRA: ...

    def first(self, arg: Structure, /) -> tuple[Model, CRA]: ...

    def set_residue_flags(self, arg: str, /) -> Selection: ...

    def set_atom_flags(self, arg: str, /) -> Selection: ...

    def copy_model_selection(self, arg: Model, /) -> Model: ...

    def copy_structure_selection(self, arg: Structure, /) -> Structure: ...

    @overload
    def remove_selected(self, arg: Structure, /) -> None: ...

    @overload
    def remove_selected(self, arg: Model, /) -> None: ...

    @overload
    def remove_not_selected(self, arg: Structure, /) -> None: ...

    @overload
    def remove_not_selected(self, arg: Model, /) -> None: ...

    def __repr__(self) -> str: ...

    def str(self) -> str: ...

class HowToNameCopiedChain(enum.Enum):
    Short = 0

    AddNumber = 1

    Dup = 2

def make_address(arg0: Chain, arg1: Residue, arg2: Atom, /) -> AtomAddress: ...

class CoorFormat(enum.Enum):
    Unknown = 0

    Detect = 1

    Pdb = 2

    Mmcif = 3

    Mmjson = 4

    ChemComp = 5

class InfoMap:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: InfoMap) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: dict[str, str], /) -> None:
        """Construct from a dictionary"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the map is nonempty"""

    def __repr__(self) -> str: ...

    @overload
    def __contains__(self, arg: str, /) -> bool: ...

    @overload
    def __contains__(self, arg: object, /) -> bool: ...

    def __iter__(self) -> Iterator[str]: ...

    def __getitem__(self, arg: str, /) -> str: ...

    def __delitem__(self, arg: str, /) -> None: ...

    def clear(self) -> None:
        """Remove all items"""

    def __setitem__(self, arg0: str, arg1: str, /) -> None: ...

    def update(self, arg: InfoMap, /) -> None:
        """Update the map with element from `arg`"""

    def __eq__(self, arg: object, /) -> bool: ...

    def __ne__(self, arg: object, /) -> bool: ...

    class ItemView:
        def __len__(self) -> int: ...

        def __iter__(self) -> Iterator[tuple[str, str]]: ...

    class KeyView:
        @overload
        def __contains__(self, arg: str, /) -> bool: ...

        @overload
        def __contains__(self, arg: object, /) -> bool: ...

        def __len__(self) -> int: ...

        def __iter__(self) -> Iterator[str]: ...

    class ValueView:
        def __len__(self) -> int: ...

        def __iter__(self) -> Iterator[str]: ...

    def keys(self) -> InfoMap.KeyView:
        """Returns an iterable view of the map's keys."""

    def values(self) -> InfoMap.ValueView:
        """Returns an iterable view of the map's values."""

    def items(self) -> InfoMap.ItemView:
        """Returns an iterable view of the map's items."""

class CRA:
    @property
    def chain(self) -> Chain: ...

    @property
    def residue(self) -> Residue: ...

    @property
    def atom(self) -> Atom: ...

    def atom_matches(self, arg: AtomAddress, /) -> bool: ...

    def __str__(self) -> str: ...

    def __repr__(self) -> str: ...

class CraGenerator:
    def __iter__(self) -> Iterator[CRA]: ...

class Structure:
    def __init__(self) -> None: ...

    @property
    def name(self) -> str: ...

    @name.setter
    def name(self, arg: str, /) -> None: ...

    @property
    def cell(self) -> UnitCell: ...

    @cell.setter
    def cell(self, arg: UnitCell, /) -> None: ...

    @property
    def spacegroup_hm(self) -> str: ...

    @spacegroup_hm.setter
    def spacegroup_hm(self, arg: str, /) -> None: ...

    @property
    def ncs(self) -> NcsOpList: ...

    @ncs.setter
    def ncs(self, arg: NcsOpList, /) -> None: ...

    @property
    def resolution(self) -> float: ...

    @resolution.setter
    def resolution(self, arg: float, /) -> None: ...

    @property
    def input_format(self) -> CoorFormat: ...

    @input_format.setter
    def input_format(self, arg: CoorFormat, /) -> None: ...

    @property
    def entities(self) -> EntityList: ...

    @entities.setter
    def entities(self, arg: EntityList, /) -> None: ...

    @property
    def connections(self) -> ConnectionList: ...

    @connections.setter
    def connections(self, arg: ConnectionList, /) -> None: ...

    @property
    def cispeps(self) -> list[CisPep]: ...

    @cispeps.setter
    def cispeps(self, arg: Sequence[CisPep], /) -> None: ...

    @property
    def mod_residues(self) -> list[ModRes]: ...

    @mod_residues.setter
    def mod_residues(self, arg: Sequence[ModRes], /) -> None: ...

    @property
    def helices(self) -> HelixList: ...

    @helices.setter
    def helices(self, arg: HelixList, /) -> None: ...

    @property
    def sheets(self) -> SheetList: ...

    @sheets.setter
    def sheets(self, arg: SheetList, /) -> None: ...

    @property
    def assemblies(self) -> AssemblyList: ...

    @assemblies.setter
    def assemblies(self, arg: AssemblyList, /) -> None: ...

    @property
    def meta(self) -> Metadata: ...

    @meta.setter
    def meta(self, arg: Metadata, /) -> None: ...

    @property
    def has_d_fraction(self) -> bool: ...

    @has_d_fraction.setter
    def has_d_fraction(self, arg: bool, /) -> None: ...

    @property
    def has_origx(self) -> bool: ...

    @has_origx.setter
    def has_origx(self, arg: bool, /) -> None: ...

    @property
    def origx(self) -> Transform: ...

    @property
    def info(self) -> InfoMap: ...

    @info.setter
    def info(self, arg: InfoMap, /) -> None: ...

    @property
    def raw_remarks(self) -> list[str]: ...

    @raw_remarks.setter
    def raw_remarks(self, arg: Sequence[str], /) -> None: ...

    def find_spacegroup(self) -> SpaceGroup: ...

    def get_entity(self, subchain: str) -> Entity: ...

    def get_entity_of(self, subchain: ResidueSpan) -> Entity: ...

    def __len__(self) -> int: ...

    def __iter__(self) -> Iterator[Model]: ...

    def __getitem__(self, index: int) -> Model: ...

    @overload
    def __delitem__(self, index: int) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

    def __setitem__(self, arg0: int, arg1: Model, /) -> None: ...

    def find_connection_by_cra(self, cra1: CRA, cra2: CRA, ignore_segment: bool = False) -> Connection: ...

    def find_connection(self, partner1: AtomAddress, partner2: AtomAddress) -> Connection: ...

    def find_or_add_model(self, name: int) -> Model: ...

    def add_model(self, model: Model, pos: int = -1) -> Model: ...

    def renumber_models(self) -> None: ...

    def merge_chain_parts(self, min_sep: int = 0) -> None: ...

    def remove_empty_chains(self) -> None: ...

    def setup_cell_images(self) -> None: ...

    def add_entity_types(self, overwrite: bool = False) -> None: ...

    def add_entity_ids(self, overwrite: bool = False) -> None: ...

    def add_conect(self, serial1: int, serial2: int, order: int) -> None: ...

    def clear_conect(self) -> None: ...

    @property
    def conect_map(self) -> dict[int, list[int]]: ...

    def assign_subchains(self, force: bool = False, fail_if_unknown: bool = True) -> None: ...

    def ensure_entities(self) -> None: ...

    def deduplicate_entities(self) -> None: ...

    def setup_entities(self) -> None: ...

    def assign_het_flags(self, flag: str = 'R') -> None: ...

    def remove_waters(self) -> None: ...

    def remove_ligands_and_waters(self) -> None: ...

    def shorten_ccd_codes(self) -> None: ...

    def restore_full_ccd_codes(self) -> None: ...

    @property
    def shortened_ccd_codes(self) -> list[tuple[str, str]]: ...

    @shortened_ccd_codes.setter
    def shortened_ccd_codes(self, arg: Sequence[tuple[str, str]], /) -> None: ...

    def remove_alternative_conformations(self) -> None: ...

    def remove_hydrogens(self) -> None: ...

    def store_deuterium_as_fraction(self, arg: bool, /) -> None: ...

    def standardize_crystal_frame(self) -> None: ...

    def assign_serial_numbers(self, numbered_ter: bool = False) -> None: ...

    def rename_chain(self, arg0: str, arg1: str, /) -> None: ...

    def rename_residues(self, arg0: str, arg1: str, /) -> None: ...

    def shorten_chain_names(self) -> None: ...

    def expand_ncs(self, how: HowToNameCopiedChain, merge_dist: float = 0.2) -> None: ...

    def transform_to_assembly(self, assembly_name: str, how: HowToNameCopiedChain, logging: object | None = None, keep_spacegroup: bool = False, merge_dist: float = 0.2) -> None: ...

    def calculate_box(self, margin: float = 0.0) -> PositionBox: ...

    def calculate_fractional_box(self, margin: float = 0.0) -> FractionalBox: ...

    def clone(self) -> Structure: ...

    def __getstate__(self) -> bytes: ...

    def __setstate__(self, arg: bytes, /) -> None: ...

    def __repr__(self) -> str: ...

    def assign_label_seq_id(self, force: bool = False) -> None: ...

    def clear_sequences(self) -> None: ...

    def assign_best_sequences(self, fasta_sequences: Sequence[str]) -> None: ...

    def make_pdb_string(self, options: PdbWriteOptions = PdbWriteOptions()) -> str: ...

    @overload
    def write_pdb(self, arg0: str, arg1: PdbWriteOptions, /) -> None: ...

    @overload
    def write_pdb(self, path: str, **kwargs) -> None: ...

    def make_pdb_headers(self) -> str: ...

    def write_minimal_pdb(self, path: str) -> None: ...

    def make_minimal_pdb(self) -> str: ...

    def make_mmcif_document(self, groups: MmcifOutputGroups = MmcifOutputGroups(True)) -> cif.Document: ...

    def make_mmcif_block(self, groups: MmcifOutputGroups = MmcifOutputGroups(True)) -> cif.Block: ...

    def update_mmcif_block(self, block: cif.Block, groups: MmcifOutputGroups = MmcifOutputGroups(True)) -> None: ...

    def make_mmcif_headers(self) -> cif.Block: ...

class MmcifOutputGroups:
    @overload
    def __init__(self, arg: bool, /) -> None: ...

    @overload
    def __init__(self, all: bool, **kwargs) -> None: ...

    @property
    def atoms(self) -> bool: ...

    @atoms.setter
    def atoms(self, arg: bool, /) -> None: ...

    @property
    def block_name(self) -> bool: ...

    @block_name.setter
    def block_name(self, arg: bool, /) -> None: ...

    @property
    def entry(self) -> bool: ...

    @entry.setter
    def entry(self, arg: bool, /) -> None: ...

    @property
    def database_status(self) -> bool: ...

    @database_status.setter
    def database_status(self, arg: bool, /) -> None: ...

    @property
    def author(self) -> bool: ...

    @author.setter
    def author(self, arg: bool, /) -> None: ...

    @property
    def cell(self) -> bool: ...

    @cell.setter
    def cell(self, arg: bool, /) -> None: ...

    @property
    def symmetry(self) -> bool: ...

    @symmetry.setter
    def symmetry(self, arg: bool, /) -> None: ...

    @property
    def entity(self) -> bool: ...

    @entity.setter
    def entity(self, arg: bool, /) -> None: ...

    @property
    def entity_poly(self) -> bool: ...

    @entity_poly.setter
    def entity_poly(self, arg: bool, /) -> None: ...

    @property
    def struct_ref(self) -> bool: ...

    @struct_ref.setter
    def struct_ref(self, arg: bool, /) -> None: ...

    @property
    def chem_comp(self) -> bool: ...

    @chem_comp.setter
    def chem_comp(self, arg: bool, /) -> None: ...

    @property
    def exptl(self) -> bool: ...

    @exptl.setter
    def exptl(self, arg: bool, /) -> None: ...

    @property
    def diffrn(self) -> bool: ...

    @diffrn.setter
    def diffrn(self, arg: bool, /) -> None: ...

    @property
    def reflns(self) -> bool: ...

    @reflns.setter
    def reflns(self, arg: bool, /) -> None: ...

    @property
    def refine(self) -> bool: ...

    @refine.setter
    def refine(self, arg: bool, /) -> None: ...

    @property
    def title_keywords(self) -> bool: ...

    @title_keywords.setter
    def title_keywords(self, arg: bool, /) -> None: ...

    @property
    def ncs(self) -> bool: ...

    @ncs.setter
    def ncs(self, arg: bool, /) -> None: ...

    @property
    def struct_asym(self) -> bool: ...

    @struct_asym.setter
    def struct_asym(self, arg: bool, /) -> None: ...

    @property
    def origx(self) -> bool: ...

    @origx.setter
    def origx(self, arg: bool, /) -> None: ...

    @property
    def struct_conf(self) -> bool: ...

    @struct_conf.setter
    def struct_conf(self, arg: bool, /) -> None: ...

    @property
    def struct_sheet(self) -> bool: ...

    @struct_sheet.setter
    def struct_sheet(self, arg: bool, /) -> None: ...

    @property
    def struct_biol(self) -> bool: ...

    @struct_biol.setter
    def struct_biol(self, arg: bool, /) -> None: ...

    @property
    def assembly(self) -> bool: ...

    @assembly.setter
    def assembly(self, arg: bool, /) -> None: ...

    @property
    def conn(self) -> bool: ...

    @conn.setter
    def conn(self, arg: bool, /) -> None: ...

    @property
    def cis(self) -> bool: ...

    @cis.setter
    def cis(self, arg: bool, /) -> None: ...

    @property
    def scale(self) -> bool: ...

    @scale.setter
    def scale(self, arg: bool, /) -> None: ...

    @property
    def atom_type(self) -> bool: ...

    @atom_type.setter
    def atom_type(self, arg: bool, /) -> None: ...

    @property
    def entity_poly_seq(self) -> bool: ...

    @entity_poly_seq.setter
    def entity_poly_seq(self, arg: bool, /) -> None: ...

    @property
    def tls(self) -> bool: ...

    @tls.setter
    def tls(self, arg: bool, /) -> None: ...

    @property
    def software(self) -> bool: ...

    @software.setter
    def software(self, arg: bool, /) -> None: ...

    @property
    def group_pdb(self) -> bool: ...

    @group_pdb.setter
    def group_pdb(self, arg: bool, /) -> None: ...

    @property
    def auth_all(self) -> bool: ...

    @auth_all.setter
    def auth_all(self, arg: bool, /) -> None: ...

class PdbWriteOptions:
    @overload
    def __init__(self, minimal: bool = False, headers_only: bool = False) -> None: ...

    @overload
    def __init__(self, minimal: bool = False, headers_only: bool = False, **kwargs) -> None: ...

    @property
    def minimal_file(self) -> bool: ...

    @minimal_file.setter
    def minimal_file(self, arg: bool, /) -> None: ...

    @property
    def atom_records(self) -> bool: ...

    @atom_records.setter
    def atom_records(self, arg: bool, /) -> None: ...

    @property
    def seqres_records(self) -> bool: ...

    @seqres_records.setter
    def seqres_records(self, arg: bool, /) -> None: ...

    @property
    def ssbond_records(self) -> bool: ...

    @ssbond_records.setter
    def ssbond_records(self, arg: bool, /) -> None: ...

    @property
    def link_records(self) -> bool: ...

    @link_records.setter
    def link_records(self, arg: bool, /) -> None: ...

    @property
    def cispep_records(self) -> bool: ...

    @cispep_records.setter
    def cispep_records(self, arg: bool, /) -> None: ...

    @property
    def cryst1_record(self) -> bool: ...

    @cryst1_record.setter
    def cryst1_record(self, arg: bool, /) -> None: ...

    @property
    def ter_records(self) -> bool: ...

    @ter_records.setter
    def ter_records(self, arg: bool, /) -> None: ...

    @property
    def conect_records(self) -> bool: ...

    @conect_records.setter
    def conect_records(self, arg: bool, /) -> None: ...

    @property
    def end_record(self) -> bool: ...

    @end_record.setter
    def end_record(self, arg: bool, /) -> None: ...

    @property
    def numbered_ter(self) -> bool: ...

    @numbered_ter.setter
    def numbered_ter(self, arg: bool, /) -> None: ...

    @property
    def ter_ignores_type(self) -> bool: ...

    @ter_ignores_type.setter
    def ter_ignores_type(self, arg: bool, /) -> None: ...

    @property
    def use_linkr(self) -> bool: ...

    @use_linkr.setter
    def use_linkr(self, arg: bool, /) -> None: ...

    @property
    def use_link_id(self) -> bool: ...

    @use_link_id.setter
    def use_link_id(self, arg: bool, /) -> None: ...

    @property
    def preserve_serial(self) -> bool: ...

    @preserve_serial.setter
    def preserve_serial(self, arg: bool, /) -> None: ...

class FirstConformerRes:
    def __iter__(self) -> Iterator[Residue]: ...

class FirstConformerResSpan:
    def __iter__(self) -> Iterator[Residue]: ...

class FirstConformerAtoms:
    def __iter__(self) -> Iterator[Atom]: ...

class ResidueSpanGroups:
    def __iter__(self) -> Iterator[ResidueGroup]: ...

class AtomGroup:
    def __len__(self) -> int: ...

    def __iter__(self) -> Iterator[Atom]: ...

    def __bool__(self) -> bool: ...

    @overload
    def __getitem__(self, index: int) -> Atom: ...

    @overload
    def __getitem__(self, altloc: str) -> Atom: ...

    def name(self) -> str: ...

    def __repr__(self) -> str: ...

class CalcFlag(enum.Enum):
    NotSet = 0

    NoHydrogen = 1

    Determined = 2

    Calculated = 3

    Dummy = 4

def calculate_b_est(arg: Atom, /) -> float: ...

def calculate_angle(arg0: Position, arg1: Position, arg2: Position, /) -> float:
    """Input: three points. Output: angle in radians."""

def calculate_dihedral(arg0: Position, arg1: Position, arg2: Position, arg3: Position, /) -> float:
    """Input: four points. Output: dihedral angle in radians."""

def calculate_omega(residue: Residue, next_residue: Residue) -> float: ...

def calculate_phi_psi(prev_residue: Residue | None, residue: Residue, next_residue: Residue | None) -> list[float]: ...

def find_best_plane(atoms: Sequence[Atom]) -> list[float]: ...

def get_distance_from_plane(pos: Position, coeff: Sequence[float]) -> float: ...

def parse_triplet_as_ftransform(arg: str, /) -> FTransform: ...

def calculate_u_from_tls(arg0: TlsGroup, arg1: Position, /) -> SMat33d: ...

def make_assembly(assembly: Assembly, model: Model, how: HowToNameCopiedChain, logging: object | None = None) -> Model: ...

def expand_ncs_model(arg0: Model, arg1: NcsOpList, arg2: HowToNameCopiedChain, /) -> Model: ...

def merge_atoms_in_expanded_model(model: Model, cell: UnitCell, max_dist: float = 0.2, compare_serial: bool = True) -> None: ...

class SelectionModelsProxy:
    def __iter__(self) -> Iterator[Model]: ...

class SelectionChainsProxy:
    def __iter__(self) -> Iterator[Chain]: ...

class SelectionResiduesProxy:
    def __iter__(self) -> Iterator[Residue]: ...

class SelectionAtomsProxy:
    def __iter__(self) -> Iterator[Atom]: ...

class SmallStructure:
    def __init__(self) -> None: ...

    class Site:
        @overload
        def __init__(self) -> None: ...

        @overload
        def __init__(self, arg0: Atom, arg1: UnitCell, /) -> None: ...

        @property
        def label(self) -> str: ...

        @label.setter
        def label(self, arg: str, /) -> None: ...

        @property
        def type_symbol(self) -> str: ...

        @type_symbol.setter
        def type_symbol(self, arg: str, /) -> None: ...

        @property
        def fract(self) -> Fractional: ...

        @fract.setter
        def fract(self, arg: Fractional, /) -> None: ...

        @property
        def occ(self) -> float: ...

        @occ.setter
        def occ(self, arg: float, /) -> None: ...

        @property
        def u_iso(self) -> float: ...

        @u_iso.setter
        def u_iso(self, arg: float, /) -> None: ...

        @property
        def element(self) -> Element: ...

        @element.setter
        def element(self, arg: Element, /) -> None: ...

        @property
        def charge(self) -> int: ...

        @charge.setter
        def charge(self, arg: int, /) -> None: ...

        @property
        def disorder_group(self) -> int: ...

        @disorder_group.setter
        def disorder_group(self, arg: int, /) -> None: ...

        @property
        def aniso(self) -> SMat33d: ...

        @aniso.setter
        def aniso(self, arg: SMat33d, /) -> None: ...

        def orth(self, arg: UnitCell, /) -> Position: ...

        def clone(self) -> SmallStructure.Site: ...

        def __repr__(self) -> str: ...

    class SiteList:
        @overload
        def __init__(self) -> None:
            """Default constructor"""

        @overload
        def __init__(self, arg: SmallStructure.SiteList) -> None:
            """Copy constructor"""

        @overload
        def __init__(self, arg: Iterable[SmallStructure.Site], /) -> None:
            """Construct from an iterable object"""

        def __len__(self) -> int: ...

        def __bool__(self) -> bool:
            """Check whether the vector is nonempty"""

        def __repr__(self) -> str: ...

        def __iter__(self) -> Iterator[SmallStructure.Site]: ...

        @overload
        def __getitem__(self, arg: int, /) -> SmallStructure.Site: ...

        @overload
        def __getitem__(self, arg: slice, /) -> SmallStructure.SiteList: ...

        def clear(self) -> None:
            """Remove all items from list."""

        def append(self, arg: SmallStructure.Site, /) -> None:
            """Append `arg` to the end of the list."""

        def insert(self, arg0: int, arg1: SmallStructure.Site, /) -> None:
            """Insert object `arg1` before index `arg0`."""

        def pop(self, index: int = -1) -> SmallStructure.Site:
            """Remove and return item at `index` (default last)."""

        def extend(self, arg: SmallStructure.SiteList, /) -> None:
            """Extend `self` by appending elements from `arg`."""

        @overload
        def __setitem__(self, arg0: int, arg1: SmallStructure.Site, /) -> None: ...

        @overload
        def __setitem__(self, arg0: slice, arg1: SmallStructure.SiteList, /) -> None: ...

        @overload
        def __delitem__(self, arg: int, /) -> None: ...

        @overload
        def __delitem__(self, arg: slice, /) -> None: ...

    class AtomType:
        @property
        def symbol(self) -> str: ...

        @property
        def element(self) -> Element: ...

        @property
        def dispersion_real(self) -> float: ...

        @dispersion_real.setter
        def dispersion_real(self, arg: float, /) -> None: ...

        @property
        def dispersion_imag(self) -> float: ...

        @dispersion_imag.setter
        def dispersion_imag(self, arg: float, /) -> None: ...

        def __repr__(self) -> str: ...

    @property
    def name(self) -> str: ...

    @name.setter
    def name(self, arg: str, /) -> None: ...

    @property
    def cell(self) -> UnitCell: ...

    @cell.setter
    def cell(self, arg: UnitCell, /) -> None: ...

    @property
    def spacegroup(self) -> SpaceGroup: ...

    @property
    def spacegroup_hm(self) -> str: ...

    @spacegroup_hm.setter
    def spacegroup_hm(self, arg: str, /) -> None: ...

    @property
    def spacegroup_hall(self) -> str: ...

    @spacegroup_hall.setter
    def spacegroup_hall(self, arg: str, /) -> None: ...

    @property
    def spacegroup_number(self) -> int: ...

    @spacegroup_number.setter
    def spacegroup_number(self, arg: int, /) -> None: ...

    @property
    def symops(self) -> list[str]: ...

    @symops.setter
    def symops(self, arg: Sequence[str], /) -> None: ...

    @property
    def sites(self) -> SmallStructure.SiteList: ...

    @sites.setter
    def sites(self, arg: SmallStructure.SiteList, /) -> None: ...

    @property
    def atom_types(self) -> list[SmallStructure.AtomType]: ...

    @property
    def wavelength(self) -> float: ...

    @wavelength.setter
    def wavelength(self, arg: float, /) -> None: ...

    def add_site(self, arg: SmallStructure.Site, /) -> None: ...

    def determine_and_set_spacegroup(self, order: str) -> None: ...

    def check_spacegroup(self) -> str: ...

    def get_atom_type(self, arg: str, /) -> SmallStructure.AtomType: ...

    def get_all_unit_cell_sites(self) -> SmallStructure.SiteList: ...

    def remove_hydrogens(self) -> None: ...

    def change_occupancies_to_crystallographic(self, max_dist: float = 0.4) -> None: ...

    def make_cif_block(self) -> cif.Block: ...

    def __repr__(self) -> str: ...

def mx_to_sx_structure(st: Structure, n: int = 0) -> SmallStructure: ...

class CifWalk:
    def __init__(self, path: str, try_pdbid: str = '\x00') -> None: ...

    def __iter__(self) -> Iterator[str]: ...

class CoorFileWalk:
    def __init__(self, path: str, try_pdbid: str = '\x00') -> None: ...

    def __iter__(self) -> Iterator[str]: ...

def is_pdb_code(arg: str, /) -> bool: ...

def expand_pdb_code_to_path(code: str, filetype: str, throw_if_unset: bool = False) -> str: ...

def expand_if_pdb_code(code: str, filetype: str = 'M') -> str: ...

hc: float = 12398.4197386209

def bessel_i1_over_i0(arg: Annotated[NDArray[numpy.float64], dict(shape=(None,), device='cpu')], /) -> NDArray[numpy.float64]: ...

def log_bessel_i0(arg: Annotated[NDArray[numpy.float64], dict(shape=(None,), device='cpu')], /) -> NDArray[numpy.float64]: ...

def log_cosh(arg: Annotated[NDArray[numpy.float64], dict(shape=(None,), device='cpu')], /) -> NDArray[numpy.float64]: ...

class FastaSeq:
    @property
    def header(self) -> str: ...

    @property
    def seq(self) -> str: ...

def read_pir_or_fasta(arg: str, /) -> list[FastaSeq]: ...

def calculate_sequence_weight(sequence: Sequence[str], unknown: float = 0.0) -> float: ...

def one_letter_code(arg: Sequence[str], /) -> str: ...

def pdbx_one_letter_code(arg0: Sequence[str], arg1: ResidueKind, /) -> str: ...

def sequence_kind(arg: PolymerType, /) -> ResidueKind: ...

class Correlation:
    @property
    def n(self) -> int: ...

    def coefficient(self) -> float: ...

    def mean_ratio(self) -> float: ...

def binmean(nbins: Annotated[NDArray[numpy.int32], dict(shape=(None,), device='cpu')], values: Annotated[NDArray[numpy.float64], dict(shape=(None,), device='cpu')]) -> NDArray[numpy.float64]: ...

def binrfactor(nbins: Annotated[NDArray[numpy.int32], dict(shape=(None,), device='cpu')], obs: Annotated[NDArray[numpy.float64], dict(shape=(None,), device='cpu')], calc: Annotated[NDArray[numpy.float64], dict(shape=(None,), device='cpu')], riso: bool = False) -> NDArray[numpy.float64]: ...

def bincorr(nbins: Annotated[NDArray[numpy.int32], dict(shape=(None,), device='cpu')], obs: Annotated[NDArray[numpy.float64], dict(shape=(None,), device='cpu')], calc: Annotated[NDArray[numpy.float64], dict(shape=(None,), device='cpu')]) -> list[Correlation]: ...

class AxisOrder(enum.Enum):
    Unknown = 0

    XYZ = 1

    ZYX = 2

class GridSizeRounding(enum.Enum):
    Nearest = 0

    Up = 1

    Down = 2

class GridMeta:
    @property
    def spacegroup(self) -> SpaceGroup: ...

    @spacegroup.setter
    def spacegroup(self, arg: SpaceGroup, /) -> None: ...

    @property
    def unit_cell(self) -> UnitCell: ...

    @unit_cell.setter
    def unit_cell(self, arg: UnitCell, /) -> None: ...

    @property
    def nu(self) -> int:
        """size in the first (fastest-changing) dim"""

    @property
    def nv(self) -> int:
        """size in the second dimension"""

    @property
    def nw(self) -> int:
        """size in the third (slowest-changing) dim"""

    @property
    def axis_order(self) -> AxisOrder: ...

    @property
    def point_count(self) -> int: ...

    def get_position(self, arg0: int, arg1: int, arg2: int, /) -> Position: ...

    def get_fractional(self, arg0: int, arg1: int, arg2: int, /) -> Fractional: ...

    @property
    def shape(self) -> tuple: ...

class Int8GridBase(GridMeta):
    class Point:
        @property
        def u(self) -> int: ...

        @property
        def v(self) -> int: ...

        @property
        def w(self) -> int: ...

        @property
        def value(self) -> int: ...

        @value.setter
        def value(self, arg: int, /) -> None: ...

        def __repr__(self) -> str: ...

    @property
    def array(self) -> NDArray[numpy.int8]: ...

    def __array__(self, dtype: object | None = None, copy: object | None = None) -> object: ...

    def point_to_index(self, arg: Int8GridBase.Point, /) -> int: ...

    def index_to_point(self, arg: int, /) -> Int8GridBase.Point: ...

    def fill(self, value: int) -> None: ...

    def sum(self) -> int: ...

    def __iter__(self) -> Iterator[Int8GridBase.Point]: ...

    def get_nonzero_extent(self) -> FractionalBox: ...

class Int8Grid(Int8GridBase):
    @overload
    def __init__(self) -> None: ...

    @overload
    def __init__(self, nx: int, ny: int, nz: int) -> None: ...

    @overload
    def __init__(self, arg0: Annotated[NDArray[numpy.int8], dict(shape=(None, None, None))], cell: UnitCell | None = None, spacegroup: SpaceGroup | None = None) -> None: ...

    @property
    def spacing(self) -> tuple: ...

    def set_size(self, arg0: int, arg1: int, arg2: int, /) -> None: ...

    def set_size_from_spacing(self, spacing: float, rounding: GridSizeRounding) -> None: ...

    def get_value(self, arg0: int, arg1: int, arg2: int, /) -> int: ...

    def set_value(self, arg0: int, arg1: int, arg2: int, arg3: int, /) -> None: ...

    def get_point(self, arg0: int, arg1: int, arg2: int, /) -> Int8GridBase.Point: ...

    def get_nearest_point(self, arg: Position, /) -> Int8GridBase.Point: ...

    def point_to_fractional(self, arg: Int8GridBase.Point, /) -> Fractional: ...

    def point_to_position(self, arg: Int8GridBase.Point, /) -> Position: ...

    def change_values(self, old_value: int, new_value: int) -> None: ...

    def copy_metadata_from(self, arg: GridMeta, /) -> None: ...

    def setup_from(self, st: Structure, spacing: float = 0.0) -> None: ...

    def set_unit_cell(self, arg: UnitCell, /) -> None: ...

    def set_points_around(self, position: Position, radius: float, value: int, use_pbc: bool = True) -> None: ...

    def symmetrize_min(self) -> None: ...

    def symmetrize_max(self) -> None: ...

    def symmetrize_abs_max(self) -> None: ...

    def symmetrize_sum(self) -> None: ...

    def masked_asu(self) -> MaskedInt8Grid: ...

    def mask_points_in_constant_radius(self, model: Model, radius: float, value: int, ignore_hydrogen: bool = False, ignore_zero_occupancy_atoms: bool = False) -> None: ...

    def get_subarray(self, start: Sequence[int], shape: Sequence[int]) -> NDArray[numpy.int8]: ...

    def set_subarray(self, arr: Annotated[NDArray[numpy.int8], dict(shape=(None, None, None), order='F', device='cpu')], start: Sequence[int]) -> None: ...

    def clone(self) -> Int8Grid: ...

    def __repr__(self) -> str: ...

class MaskedInt8Grid:
    @property
    def grid(self) -> Int8Grid: ...

    @property
    def mask_array(self) -> NDArray[numpy.int8]: ...

    def __iter__(self) -> Iterator[Int8GridBase.Point]: ...

class FloatGridBase(GridMeta):
    class Point:
        @property
        def u(self) -> int: ...

        @property
        def v(self) -> int: ...

        @property
        def w(self) -> int: ...

        @property
        def value(self) -> float: ...

        @value.setter
        def value(self, arg: float, /) -> None: ...

        def __repr__(self) -> str: ...

    @property
    def array(self) -> NDArray[numpy.float32]: ...

    def __array__(self, dtype: object | None = None, copy: object | None = None) -> object: ...

    def point_to_index(self, arg: FloatGridBase.Point, /) -> int: ...

    def index_to_point(self, arg: int, /) -> FloatGridBase.Point: ...

    def fill(self, value: float) -> None: ...

    def sum(self) -> float: ...

    def __iter__(self) -> Iterator[FloatGridBase.Point]: ...

    def calculate_correlation(self, arg: FloatGridBase, /) -> Correlation: ...

    def get_nonzero_extent(self) -> FractionalBox: ...

class FloatGrid(FloatGridBase):
    @overload
    def __init__(self) -> None: ...

    @overload
    def __init__(self, nx: int, ny: int, nz: int) -> None: ...

    @overload
    def __init__(self, arg0: Annotated[NDArray[numpy.float32], dict(shape=(None, None, None))], cell: UnitCell | None = None, spacegroup: SpaceGroup | None = None) -> None: ...

    @property
    def spacing(self) -> tuple: ...

    def set_size(self, arg0: int, arg1: int, arg2: int, /) -> None: ...

    def set_size_from_spacing(self, spacing: float, rounding: GridSizeRounding) -> None: ...

    def get_value(self, arg0: int, arg1: int, arg2: int, /) -> float: ...

    def set_value(self, arg0: int, arg1: int, arg2: int, arg3: float, /) -> None: ...

    def get_point(self, arg0: int, arg1: int, arg2: int, /) -> FloatGridBase.Point: ...

    def get_nearest_point(self, arg: Position, /) -> FloatGridBase.Point: ...

    def point_to_fractional(self, arg: FloatGridBase.Point, /) -> Fractional: ...

    def point_to_position(self, arg: FloatGridBase.Point, /) -> Position: ...

    def change_values(self, old_value: float, new_value: float) -> None: ...

    def copy_metadata_from(self, arg: GridMeta, /) -> None: ...

    def setup_from(self, st: Structure, spacing: float = 0.0) -> None: ...

    def set_unit_cell(self, arg: UnitCell, /) -> None: ...

    def set_points_around(self, position: Position, radius: float, value: float, use_pbc: bool = True) -> None: ...

    def symmetrize_min(self) -> None: ...

    def symmetrize_max(self) -> None: ...

    def symmetrize_abs_max(self) -> None: ...

    def symmetrize_sum(self) -> None: ...

    def masked_asu(self) -> MaskedFloatGrid: ...

    def mask_points_in_constant_radius(self, model: Model, radius: float, value: float, ignore_hydrogen: bool = False, ignore_zero_occupancy_atoms: bool = False) -> None: ...

    def get_subarray(self, start: Sequence[int], shape: Sequence[int]) -> NDArray[numpy.float32]: ...

    def set_subarray(self, arr: Annotated[NDArray[numpy.float32], dict(shape=(None, None, None), order='F', device='cpu')], start: Sequence[int]) -> None: ...

    def clone(self) -> FloatGrid: ...

    def __repr__(self) -> str: ...

    @overload
    def interpolate_value(self, arg0: Fractional, order: int = 1) -> float: ...

    @overload
    def interpolate_value(self, arg0: Position, order: int = 1) -> float: ...

    @overload
    def tricubic_interpolation(self, arg: Fractional, /) -> float: ...

    @overload
    def tricubic_interpolation(self, arg: Position, /) -> float: ...

    def tricubic_interpolation_der(self, arg: Fractional, /) -> list[float]: ...

    def interpolate_position_array(self, xyz: Annotated[NDArray[numpy.float64], dict(shape=(None, 3), device='cpu')], order: int = 1, to_frac: Transform | None = None) -> NDArray[numpy.float32]: ...

    def interpolate_values(self, arg0: Annotated[NDArray[numpy.float32], dict(shape=(None, None, None))], arg1: Transform, order: int = 1) -> None: ...

    def interpolate_grid_flexible(self, arg0: FloatGrid, arg1: Sequence[Annotated[NDArray[numpy.int32], dict(shape=(None, 3))]], arg2: Sequence[Annotated[NDArray[numpy.float32], dict(shape=(None, 3))]], arg3: Sequence[Transform]) -> None: ...

    def symmetrize_avg(self) -> None: ...

    def normalize(self) -> None: ...

    def add_soft_edge_to_mask(self, arg: float, /) -> None: ...

class MaskedFloatGrid:
    @property
    def grid(self) -> FloatGrid: ...

    @property
    def mask_array(self) -> NDArray[numpy.int8]: ...

    def __iter__(self) -> Iterator[FloatGridBase.Point]: ...

class ComplexGridBase(GridMeta):
    class Point:
        @property
        def u(self) -> int: ...

        @property
        def v(self) -> int: ...

        @property
        def w(self) -> int: ...

        @property
        def value(self) -> complex: ...

        @value.setter
        def value(self, arg: complex, /) -> None: ...

        def __repr__(self) -> str: ...

    @property
    def array(self) -> NDArray[numpy.complex64]: ...

    def __array__(self, dtype: object | None = None, copy: object | None = None) -> object: ...

    def point_to_index(self, arg: ComplexGridBase.Point, /) -> int: ...

    def index_to_point(self, arg: int, /) -> ComplexGridBase.Point: ...

    def fill(self, value: complex) -> None: ...

    def sum(self) -> complex: ...

    def __iter__(self) -> Iterator[ComplexGridBase.Point]: ...

class AtomicRadiiSet(enum.Enum):
    VanDerWaals = 0

    Cctbx = 1

    Refmac = 2

    Constant = 3

class SolventMasker:
    def __init__(self, choice: AtomicRadiiSet, constant_r: float = 0.0) -> None: ...

    @property
    def atomic_radii_set(self) -> AtomicRadiiSet: ...

    @atomic_radii_set.setter
    def atomic_radii_set(self, arg: AtomicRadiiSet, /) -> None: ...

    @property
    def rprobe(self) -> float: ...

    @rprobe.setter
    def rprobe(self, arg: float, /) -> None: ...

    @property
    def rshrink(self) -> float: ...

    @rshrink.setter
    def rshrink(self, arg: float, /) -> None: ...

    @property
    def island_min_volume(self) -> float: ...

    @island_min_volume.setter
    def island_min_volume(self, arg: float, /) -> None: ...

    @property
    def constant_r(self) -> float: ...

    @constant_r.setter
    def constant_r(self, arg: float, /) -> None: ...

    @property
    def ignore_hydrogen(self) -> bool: ...

    @ignore_hydrogen.setter
    def ignore_hydrogen(self, arg: bool, /) -> None: ...

    @property
    def ignore_zero_occupancy_atoms(self) -> bool: ...

    @ignore_zero_occupancy_atoms.setter
    def ignore_zero_occupancy_atoms(self, arg: bool, /) -> None: ...

    def set_radii(self, choice: AtomicRadiiSet, constant_r: float = 0.0) -> None: ...

    def put_mask_on_int8_grid(self, arg0: Int8Grid, arg1: Model, /) -> None: ...

    def put_mask_on_float_grid(self, arg0: FloatGrid, arg1: Model, /) -> None: ...

    def set_to_zero(self, arg0: FloatGrid, arg1: Model, /) -> None: ...

def interpolate_grid(dest: FloatGrid, src: FloatGrid, tr: Transform, order: int = 1) -> None: ...

def interpolate_grid_around_model(dest: FloatGrid, src: FloatGrid, tr: Transform, dest_model: Model, radius: float, order: int = 1) -> None: ...

class Blob:
    @property
    def volume(self) -> float: ...

    @property
    def score(self) -> float: ...

    @property
    def peak_value(self) -> float: ...

    @property
    def centroid(self) -> Position: ...

    @property
    def peak_pos(self) -> Position: ...

def find_blobs_by_flood_fill(grid: FloatGrid, cutoff: float, min_volume: float = 10.0, min_score: float = 15.0, min_peak: float = 0.0, negate: bool = False) -> list[Blob]: ...

def flood_fill_above(grid: FloatGrid, seeds: Sequence[Position], threshold: float, negate: bool = False) -> Int8Grid: ...

class AsuBrick:
    @property
    def size(self) -> list[int]: ...

    @property
    def incl(self) -> list[bool]: ...

    def get_extent(self) -> FractionalBox: ...

    def str(self) -> str: ...

def find_asu_brick(arg: SpaceGroup, /) -> AsuBrick: ...

class ValueSigma:
    @property
    def value(self) -> float: ...

    @value.setter
    def value(self, arg: float, /) -> None: ...

    @property
    def sigma(self) -> float: ...

    @sigma.setter
    def sigma(self, arg: float, /) -> None: ...

    def __repr__(self) -> str: ...

class ComplexCorrelation:
    @property
    def n(self) -> int: ...

    def coefficient(self) -> complex: ...

    def mean_ratio(self) -> float: ...

class ReciprocalInt8Grid(Int8GridBase):
    @overload
    def __init__(self) -> None: ...

    @overload
    def __init__(self, nx: int, ny: int, nz: int) -> None: ...

    @overload
    def __init__(self, arg0: Annotated[NDArray[numpy.int8], dict(shape=(None, None, None), device='cpu')], cell: UnitCell | None = None, spacegroup: SpaceGroup | None = None) -> None: ...

    @property
    def half_l(self) -> bool: ...

    def get_value(self, arg0: int, arg1: int, arg2: int, /) -> int: ...

    def get_value_or_zero(self, arg0: int, arg1: int, arg2: int, /) -> int: ...

    def set_value(self, arg0: int, arg1: int, arg2: int, arg3: int, /) -> None: ...

    def to_hkl(self, arg: Int8GridBase.Point, /) -> list[int]: ...

    def calculate_1_d2(self, arg: Int8GridBase.Point, /) -> float: ...

    def calculate_d(self, arg: Int8GridBase.Point, /) -> float: ...

    def get_value_by_hkl(self, hkl: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), device='cpu')], unblur: float = 0, mott_bethe: bool = False, mott_bethe_000: int = 0) -> NDArray[numpy.int32]: ...

    def prepare_asu_data(self, dmin: float = 0.0, unblur: float = 0.0, with_000: bool = False, with_sys_abs: bool = False, mott_bethe: bool = False) -> IntAsuData: ...

    def __repr__(self) -> str: ...

class IntHklValue:
    @property
    def hkl(self) -> list[int]: ...

    @property
    def value(self) -> int: ...

    @value.setter
    def value(self, arg: int, /) -> None: ...

    def __repr__(self) -> str: ...

class IntAsuData:
    def __init__(self, cell: UnitCell, sg: SpaceGroup, miller_array: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), device='cpu')], value_array: Annotated[NDArray[numpy.int32], dict(shape=(None,), order='C', device='cpu')]) -> None: ...

    def __iter__(self) -> Iterator[IntHklValue]: ...

    def __len__(self) -> int: ...

    def __getitem__(self, index: int) -> IntHklValue: ...

    @property
    def spacegroup(self) -> SpaceGroup: ...

    @spacegroup.setter
    def spacegroup(self, arg: SpaceGroup, /) -> None: ...

    @property
    def unit_cell(self) -> UnitCell: ...

    @unit_cell.setter
    def unit_cell(self, arg: UnitCell, /) -> None: ...

    @property
    def miller_array(self) -> Annotated[NDArray[numpy.int32], dict(shape=(None, 3))]: ...

    @property
    def value_array(self) -> Annotated[NDArray[numpy.int32], dict(shape=(None,))]: ...

    def make_1_d2_array(self) -> NDArray[numpy.float32]: ...

    def make_d_array(self) -> NDArray[numpy.float32]: ...

    def count_equal_values(self, arg: IntAsuData, /) -> int: ...

    def ensure_sorted(self) -> None: ...

    def ensure_asu(self, tnt_asu: bool = False) -> None: ...

    def copy(self) -> IntAsuData: ...

    def __repr__(self) -> str: ...

class ReciprocalFloatGrid(FloatGridBase):
    @overload
    def __init__(self) -> None: ...

    @overload
    def __init__(self, nx: int, ny: int, nz: int) -> None: ...

    @overload
    def __init__(self, arg0: Annotated[NDArray[numpy.float32], dict(shape=(None, None, None), device='cpu')], cell: UnitCell | None = None, spacegroup: SpaceGroup | None = None) -> None: ...

    @property
    def half_l(self) -> bool: ...

    def get_value(self, arg0: int, arg1: int, arg2: int, /) -> float: ...

    def get_value_or_zero(self, arg0: int, arg1: int, arg2: int, /) -> float: ...

    def set_value(self, arg0: int, arg1: int, arg2: int, arg3: float, /) -> None: ...

    def to_hkl(self, arg: FloatGridBase.Point, /) -> list[int]: ...

    def calculate_1_d2(self, arg: FloatGridBase.Point, /) -> float: ...

    def calculate_d(self, arg: FloatGridBase.Point, /) -> float: ...

    def get_value_by_hkl(self, hkl: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), device='cpu')], unblur: float = 0, mott_bethe: bool = False, mott_bethe_000: float = 0) -> NDArray[numpy.float32]: ...

    def prepare_asu_data(self, dmin: float = 0.0, unblur: float = 0.0, with_000: bool = False, with_sys_abs: bool = False, mott_bethe: bool = False) -> FloatAsuData: ...

    def __repr__(self) -> str: ...

class FloatHklValue:
    @property
    def hkl(self) -> list[int]: ...

    @property
    def value(self) -> float: ...

    @value.setter
    def value(self, arg: float, /) -> None: ...

    def __repr__(self) -> str: ...

class FloatAsuData:
    def __init__(self, cell: UnitCell, sg: SpaceGroup, miller_array: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), device='cpu')], value_array: Annotated[NDArray[numpy.float32], dict(shape=(None,), order='C', device='cpu')]) -> None: ...

    def __iter__(self) -> Iterator[FloatHklValue]: ...

    def __len__(self) -> int: ...

    def __getitem__(self, index: int) -> FloatHklValue: ...

    @property
    def spacegroup(self) -> SpaceGroup: ...

    @spacegroup.setter
    def spacegroup(self, arg: SpaceGroup, /) -> None: ...

    @property
    def unit_cell(self) -> UnitCell: ...

    @unit_cell.setter
    def unit_cell(self, arg: UnitCell, /) -> None: ...

    @property
    def miller_array(self) -> Annotated[NDArray[numpy.int32], dict(shape=(None, 3))]: ...

    @property
    def value_array(self) -> Annotated[NDArray[numpy.float32], dict(shape=(None,))]: ...

    def make_1_d2_array(self) -> NDArray[numpy.float32]: ...

    def make_d_array(self) -> NDArray[numpy.float32]: ...

    def count_equal_values(self, arg: FloatAsuData, /) -> int: ...

    def ensure_sorted(self) -> None: ...

    def ensure_asu(self, tnt_asu: bool = False) -> None: ...

    def copy(self) -> FloatAsuData: ...

    def __repr__(self) -> str: ...

    def calculate_correlation(self, arg: FloatAsuData, /) -> Correlation: ...

class ReciprocalComplexGrid(ComplexGridBase):
    @overload
    def __init__(self) -> None: ...

    @overload
    def __init__(self, nx: int, ny: int, nz: int) -> None: ...

    @overload
    def __init__(self, arg0: Annotated[NDArray[numpy.complex64], dict(shape=(None, None, None), device='cpu')], cell: UnitCell | None = None, spacegroup: SpaceGroup | None = None) -> None: ...

    @property
    def half_l(self) -> bool: ...

    def get_value(self, arg0: int, arg1: int, arg2: int, /) -> complex: ...

    def get_value_or_zero(self, arg0: int, arg1: int, arg2: int, /) -> complex: ...

    def set_value(self, arg0: int, arg1: int, arg2: int, arg3: complex, /) -> None: ...

    def to_hkl(self, arg: ComplexGridBase.Point, /) -> list[int]: ...

    def calculate_1_d2(self, arg: ComplexGridBase.Point, /) -> float: ...

    def calculate_d(self, arg: ComplexGridBase.Point, /) -> float: ...

    def get_value_by_hkl(self, hkl: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), device='cpu')], unblur: float = 0, mott_bethe: bool = False, mott_bethe_000: complex = 0) -> NDArray[numpy.complex64]: ...

    def prepare_asu_data(self, dmin: float = 0.0, unblur: float = 0.0, with_000: bool = False, with_sys_abs: bool = False, mott_bethe: bool = False) -> ComplexAsuData: ...

    def __repr__(self) -> str: ...

class ComplexHklValue:
    @property
    def hkl(self) -> list[int]: ...

    @property
    def value(self) -> complex: ...

    @value.setter
    def value(self, arg: complex, /) -> None: ...

    def __repr__(self) -> str: ...

class ComplexAsuData:
    def __init__(self, cell: UnitCell, sg: SpaceGroup, miller_array: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), device='cpu')], value_array: Annotated[NDArray[numpy.complex64], dict(shape=(None,), order='C', device='cpu')]) -> None: ...

    def __iter__(self) -> Iterator[ComplexHklValue]: ...

    def __len__(self) -> int: ...

    def __getitem__(self, index: int) -> ComplexHklValue: ...

    @property
    def spacegroup(self) -> SpaceGroup: ...

    @spacegroup.setter
    def spacegroup(self, arg: SpaceGroup, /) -> None: ...

    @property
    def unit_cell(self) -> UnitCell: ...

    @unit_cell.setter
    def unit_cell(self, arg: UnitCell, /) -> None: ...

    @property
    def miller_array(self) -> Annotated[NDArray[numpy.int32], dict(shape=(None, 3))]: ...

    @property
    def value_array(self) -> Annotated[NDArray[numpy.complex64], dict(shape=(None,))]: ...

    def make_1_d2_array(self) -> NDArray[numpy.float32]: ...

    def make_d_array(self) -> NDArray[numpy.float32]: ...

    def count_equal_values(self, arg: ComplexAsuData, /) -> int: ...

    def ensure_sorted(self) -> None: ...

    def ensure_asu(self, tnt_asu: bool = False) -> None: ...

    def copy(self) -> ComplexAsuData: ...

    def __repr__(self) -> str: ...

    def get_size_for_hkl(self, min_size: Sequence[int] = [0, 0, 0], sample_rate: float = 0.0) -> list[int]: ...

    def data_fits_into(self, size: Sequence[int]) -> bool: ...

    def get_f_phi_on_grid(self, size: Sequence[int], half_l: bool = False, order: AxisOrder = AxisOrder.XYZ) -> ReciprocalComplexGrid: ...

    def transform_f_phi_to_map(self, min_size: Sequence[int] = [0, 0, 0], sample_rate: float = 0.0, exact_size: Sequence[int] = [0, 0, 0], order: AxisOrder = AxisOrder.XYZ) -> FloatGrid: ...

    def calculate_correlation(self, arg: ComplexAsuData, /) -> ComplexCorrelation: ...

class ValueSigmaHklValue:
    @property
    def hkl(self) -> list[int]: ...

    @property
    def value(self) -> ValueSigma: ...

    @value.setter
    def value(self, arg: ValueSigma, /) -> None: ...

    def __repr__(self) -> str: ...

class ValueSigmaAsuData:
    def __init__(self, cell: UnitCell, sg: SpaceGroup, miller_array: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), device='cpu')], value_array: Annotated[NDArray[numpy.float32], dict(shape=(None, 2), order='C', device='cpu')]) -> None: ...

    def __iter__(self) -> Iterator[ValueSigmaHklValue]: ...

    def __len__(self) -> int: ...

    def __getitem__(self, index: int) -> ValueSigmaHklValue: ...

    @property
    def spacegroup(self) -> SpaceGroup: ...

    @spacegroup.setter
    def spacegroup(self, arg: SpaceGroup, /) -> None: ...

    @property
    def unit_cell(self) -> UnitCell: ...

    @unit_cell.setter
    def unit_cell(self, arg: UnitCell, /) -> None: ...

    @property
    def miller_array(self) -> Annotated[NDArray[numpy.int32], dict(shape=(None, 3))]: ...

    @property
    def value_array(self) -> Annotated[NDArray[numpy.float32], dict(shape=(None, 2))]: ...

    def make_1_d2_array(self) -> NDArray[numpy.float32]: ...

    def make_d_array(self) -> NDArray[numpy.float32]: ...

    def count_equal_values(self, arg: ValueSigmaAsuData, /) -> int: ...

    def ensure_sorted(self) -> None: ...

    def ensure_asu(self, tnt_asu: bool = False) -> None: ...

    def copy(self) -> ValueSigmaAsuData: ...

    def __repr__(self) -> str: ...

    def discard_by_sigma_ratio(self, arg: float, /) -> None: ...

class MapSetup(enum.Enum):
    Full = 0

    NoSymmetry = 1

    ReorderOnly = 2

class Ccp4Base:
    @property
    def ccp4_header(self) -> bytes: ...

    def header_i32(self, arg: int, /) -> int: ...

    def header_float(self, arg: int, /) -> float: ...

    def header_str(self, w: int, len: int = 80) -> str: ...

    def set_header_i32(self, arg0: int, arg1: int, /) -> None: ...

    def set_header_float(self, arg0: int, arg1: float, /) -> None: ...

    def set_header_str(self, arg0: int, arg1: str, /) -> None: ...

    def axis_positions(self) -> list[int]: ...

    def get_extent(self) -> FractionalBox: ...

    def has_skew_transformation(self) -> bool: ...

    def get_skew_transformation(self) -> Transform: ...

class Ccp4Map(Ccp4Base):
    def __init__(self) -> None: ...

    @property
    def grid(self) -> FloatGrid: ...

    @grid.setter
    def grid(self, arg: FloatGrid, /) -> None: ...

    def setup(self, default_value: float, mode: MapSetup = MapSetup.Full) -> None: ...

    def update_ccp4_header(self, mode: int = -1, update_stats: bool = True) -> None: ...

    def full_cell(self) -> bool: ...

    def write_ccp4_map(self, filename: str) -> None: ...

    def set_extent(self, arg: FractionalBox, /) -> None: ...

    def __repr__(self) -> str: ...

class Ccp4Mask(Ccp4Base):
    def __init__(self) -> None: ...

    @property
    def grid(self) -> Int8Grid: ...

    @grid.setter
    def grid(self, arg: Int8Grid, /) -> None: ...

    def setup(self, default_value: int, mode: MapSetup = MapSetup.Full) -> None: ...

    def update_ccp4_header(self, mode: int = -1, update_stats: bool = True) -> None: ...

    def full_cell(self) -> bool: ...

    def write_ccp4_map(self, filename: str) -> None: ...

    def set_extent(self, arg: FractionalBox, /) -> None: ...

    def __repr__(self) -> str: ...

def read_ccp4_map(path: str, setup: bool = False) -> Ccp4Map:
    """Reads a CCP4 file, mode 2 (floating-point data)."""

def read_ccp4_mask(path: str, setup: bool = False) -> Ccp4Mask:
    """Reads a CCP4 file, mode 0 (int8_t data, usually 0/1 masks)."""

def read_ccp4_header(path: str) -> Ccp4Base: ...

class Addends:
    def set(self, arg0: Element, arg1: float, /) -> None: ...

    def get(self, arg: Element, /) -> float: ...

    def clear(self) -> None: ...

    def add_cl_fprime(self, energy: float) -> None: ...

    def subtract_z(self, except_hydrogen: bool = False) -> None: ...

class StructureFactorCalculatorX:
    def __init__(self, arg: UnitCell, /) -> None: ...

    @property
    def addends(self) -> Addends: ...

    @addends.setter
    def addends(self, arg: Addends, /) -> None: ...

    def calculate_sf_from_model(self, arg0: Model, arg1: Sequence[int], /) -> complex: ...

    def calculate_sf_from_small_structure(self, arg0: SmallStructure, arg1: Sequence[int], /) -> complex: ...

    def mott_bethe_factor(self) -> float: ...

    def calculate_mb_z(self, model: Model, hkl: Sequence[int], only_h: bool = False) -> complex: ...

class StructureFactorCalculatorE:
    def __init__(self, arg: UnitCell, /) -> None: ...

    @property
    def addends(self) -> Addends: ...

    @addends.setter
    def addends(self, arg: Addends, /) -> None: ...

    def calculate_sf_from_model(self, arg0: Model, arg1: Sequence[int], /) -> complex: ...

    def calculate_sf_from_small_structure(self, arg0: SmallStructure, arg1: Sequence[int], /) -> complex: ...

class StructureFactorCalculatorN:
    def __init__(self, arg: UnitCell, /) -> None: ...

    @property
    def addends(self) -> Addends: ...

    @addends.setter
    def addends(self, arg: Addends, /) -> None: ...

    def calculate_sf_from_model(self, arg0: Model, arg1: Sequence[int], /) -> complex: ...

    def calculate_sf_from_small_structure(self, arg0: SmallStructure, arg1: Sequence[int], /) -> complex: ...

class DensityCalculatorX:
    def __init__(self) -> None: ...

    @property
    def grid(self) -> FloatGrid: ...

    @property
    def d_min(self) -> float: ...

    @d_min.setter
    def d_min(self, arg: float, /) -> None: ...

    @property
    def rate(self) -> float: ...

    @rate.setter
    def rate(self, arg: float, /) -> None: ...

    @property
    def blur(self) -> float: ...

    @blur.setter
    def blur(self, arg: float, /) -> None: ...

    @property
    def cutoff(self) -> float: ...

    @cutoff.setter
    def cutoff(self, arg: float, /) -> None: ...

    @property
    def addends(self) -> Addends: ...

    @addends.setter
    def addends(self, arg: Addends, /) -> None: ...

    def set_refmac_compatible_blur(self, model: Model, allow_negative: bool = False) -> None: ...

    def put_model_density_on_grid(self, arg: Model, /) -> None: ...

    def initialize_grid(self) -> None: ...

    def add_model_density_to_grid(self, arg: Model, /) -> None: ...

    def add_atom_density_to_grid(self, arg: Atom, /) -> None: ...

    def add_c_contribution_to_grid(self, arg0: Atom, arg1: float, /) -> None: ...

    def set_grid_cell_and_spacegroup(self, arg: Structure, /) -> None: ...

    def reciprocal_space_multiplier(self, arg: float, /) -> float: ...

    def mott_bethe_factor(self, arg: Sequence[int], /) -> float: ...

    def estimate_radius(self, arg: Atom, /) -> float: ...

class DensityCalculatorE:
    def __init__(self) -> None: ...

    @property
    def grid(self) -> FloatGrid: ...

    @property
    def d_min(self) -> float: ...

    @d_min.setter
    def d_min(self, arg: float, /) -> None: ...

    @property
    def rate(self) -> float: ...

    @rate.setter
    def rate(self, arg: float, /) -> None: ...

    @property
    def blur(self) -> float: ...

    @blur.setter
    def blur(self, arg: float, /) -> None: ...

    @property
    def cutoff(self) -> float: ...

    @cutoff.setter
    def cutoff(self, arg: float, /) -> None: ...

    @property
    def addends(self) -> Addends: ...

    @addends.setter
    def addends(self, arg: Addends, /) -> None: ...

    def set_refmac_compatible_blur(self, model: Model, allow_negative: bool = False) -> None: ...

    def put_model_density_on_grid(self, arg: Model, /) -> None: ...

    def initialize_grid(self) -> None: ...

    def add_model_density_to_grid(self, arg: Model, /) -> None: ...

    def add_atom_density_to_grid(self, arg: Atom, /) -> None: ...

    def add_c_contribution_to_grid(self, arg0: Atom, arg1: float, /) -> None: ...

    def set_grid_cell_and_spacegroup(self, arg: Structure, /) -> None: ...

    def reciprocal_space_multiplier(self, arg: float, /) -> float: ...

    def mott_bethe_factor(self, arg: Sequence[int], /) -> float: ...

    def estimate_radius(self, arg: Atom, /) -> float: ...

class DensityCalculatorN:
    def __init__(self) -> None: ...

    @property
    def grid(self) -> FloatGrid: ...

    @property
    def d_min(self) -> float: ...

    @d_min.setter
    def d_min(self, arg: float, /) -> None: ...

    @property
    def rate(self) -> float: ...

    @rate.setter
    def rate(self, arg: float, /) -> None: ...

    @property
    def blur(self) -> float: ...

    @blur.setter
    def blur(self, arg: float, /) -> None: ...

    @property
    def cutoff(self) -> float: ...

    @cutoff.setter
    def cutoff(self, arg: float, /) -> None: ...

    @property
    def addends(self) -> Addends: ...

    @addends.setter
    def addends(self, arg: Addends, /) -> None: ...

    def set_refmac_compatible_blur(self, model: Model, allow_negative: bool = False) -> None: ...

    def put_model_density_on_grid(self, arg: Model, /) -> None: ...

    def initialize_grid(self) -> None: ...

    def add_model_density_to_grid(self, arg: Model, /) -> None: ...

    def add_atom_density_to_grid(self, arg: Atom, /) -> None: ...

    def add_c_contribution_to_grid(self, arg0: Atom, arg1: float, /) -> None: ...

    def set_grid_cell_and_spacegroup(self, arg: Structure, /) -> None: ...

    def reciprocal_space_multiplier(self, arg: float, /) -> float: ...

    def mott_bethe_factor(self, arg: Sequence[int], /) -> float: ...

    def estimate_radius(self, arg: Atom, /) -> float: ...

class DensityCalculatorC:
    def __init__(self) -> None: ...

    @property
    def grid(self) -> FloatGrid: ...

    @property
    def d_min(self) -> float: ...

    @d_min.setter
    def d_min(self, arg: float, /) -> None: ...

    @property
    def rate(self) -> float: ...

    @rate.setter
    def rate(self, arg: float, /) -> None: ...

    @property
    def blur(self) -> float: ...

    @blur.setter
    def blur(self, arg: float, /) -> None: ...

    @property
    def cutoff(self) -> float: ...

    @cutoff.setter
    def cutoff(self, arg: float, /) -> None: ...

    @property
    def addends(self) -> Addends: ...

    @addends.setter
    def addends(self, arg: Addends, /) -> None: ...

    def set_refmac_compatible_blur(self, model: Model, allow_negative: bool = False) -> None: ...

    def put_model_density_on_grid(self, arg: Model, /) -> None: ...

    def initialize_grid(self) -> None: ...

    def add_model_density_to_grid(self, arg: Model, /) -> None: ...

    def add_atom_density_to_grid(self, arg: Atom, /) -> None: ...

    def add_c_contribution_to_grid(self, arg0: Atom, arg1: float, /) -> None: ...

    def set_grid_cell_and_spacegroup(self, arg: Structure, /) -> None: ...

    def reciprocal_space_multiplier(self, arg: float, /) -> float: ...

    def mott_bethe_factor(self, arg: Sequence[int], /) -> float: ...

    def estimate_radius(self, arg: Atom, /) -> float: ...

def mott_bethe_const() -> float: ...

class Mtz:
    def __init__(self, with_base: bool = False) -> None: ...

    class Dataset:
        @property
        def id(self) -> int: ...

        @id.setter
        def id(self, arg: int, /) -> None: ...

        @property
        def project_name(self) -> str: ...

        @project_name.setter
        def project_name(self, arg: str, /) -> None: ...

        @property
        def crystal_name(self) -> str: ...

        @crystal_name.setter
        def crystal_name(self, arg: str, /) -> None: ...

        @property
        def dataset_name(self) -> str: ...

        @dataset_name.setter
        def dataset_name(self, arg: str, /) -> None: ...

        @property
        def cell(self) -> UnitCell: ...

        @cell.setter
        def cell(self, arg: UnitCell, /) -> None: ...

        @property
        def wavelength(self) -> float: ...

        @wavelength.setter
        def wavelength(self, arg: float, /) -> None: ...

        def __repr__(self) -> str: ...

    class Column:
        @property
        def array(self) -> Annotated[NDArray[numpy.float32], dict(shape=(None,))]: ...

        def __array__(self, dtype: object | None = None, copy: object | None = None) -> object: ...

        @property
        def dataset(self) -> Mtz.Dataset: ...

        @property
        def dataset_id(self) -> int: ...

        @dataset_id.setter
        def dataset_id(self, arg: int, /) -> None: ...

        @property
        def type(self) -> str: ...

        @type.setter
        def type(self, arg: str, /) -> None: ...

        @property
        def label(self) -> str: ...

        @label.setter
        def label(self, arg: str, /) -> None: ...

        @property
        def min_value(self) -> float: ...

        @min_value.setter
        def min_value(self, arg: float, /) -> None: ...

        @property
        def max_value(self) -> float: ...

        @max_value.setter
        def max_value(self, arg: float, /) -> None: ...

        @property
        def source(self) -> str: ...

        @source.setter
        def source(self, arg: str, /) -> None: ...

        @property
        def idx(self) -> int: ...

        @idx.setter
        def idx(self, arg: int, /) -> None: ...

        def is_integer(self) -> bool: ...

        def __len__(self) -> int: ...

        def __getitem__(self, index: int) -> float: ...

        def __iter__(self) -> Iterator[float]: ...

        def __repr__(self) -> str: ...

    class Batch:
        def __init__(self) -> None: ...

        @property
        def number(self) -> int: ...

        @number.setter
        def number(self, arg: int, /) -> None: ...

        @property
        def title(self) -> str: ...

        @title.setter
        def title(self, arg: str, /) -> None: ...

        @property
        def ints(self) -> BatchInts: ...

        @ints.setter
        def ints(self, arg: BatchInts, /) -> None: ...

        @property
        def floats(self) -> BatchFloats: ...

        @floats.setter
        def floats(self, arg: BatchFloats, /) -> None: ...

        @property
        def axes(self) -> list[str]: ...

        @axes.setter
        def axes(self, arg: Sequence[str], /) -> None: ...

        @property
        def cell(self) -> UnitCell: ...

        @cell.setter
        def cell(self, arg: UnitCell, /) -> None: ...

        @property
        def dataset_id(self) -> int: ...

        @dataset_id.setter
        def dataset_id(self, arg: int, /) -> None: ...

        @property
        def wavelength(self) -> float: ...

        @wavelength.setter
        def wavelength(self, arg: float, /) -> None: ...

        def clone(self) -> Mtz.Batch: ...

    @property
    def array(self) -> Annotated[NDArray[numpy.float32], dict(shape=(None, None))]: ...

    def __array__(self, dtype: object | None = None, copy: object | None = None) -> object: ...

    @property
    def title(self) -> str: ...

    @title.setter
    def title(self, arg: str, /) -> None: ...

    @property
    def nreflections(self) -> int: ...

    @nreflections.setter
    def nreflections(self, arg: int, /) -> None: ...

    @property
    def sort_order(self) -> list[int]: ...

    @sort_order.setter
    def sort_order(self, arg: Sequence[int], /) -> None: ...

    @property
    def min_1_d2(self) -> float: ...

    @min_1_d2.setter
    def min_1_d2(self, arg: float, /) -> None: ...

    @property
    def max_1_d2(self) -> float: ...

    @max_1_d2.setter
    def max_1_d2(self, arg: float, /) -> None: ...

    @property
    def valm(self) -> float: ...

    @valm.setter
    def valm(self, arg: float, /) -> None: ...

    @property
    def nsymop(self) -> int: ...

    @property
    def cell(self) -> UnitCell: ...

    @cell.setter
    def cell(self, arg: UnitCell, /) -> None: ...

    @property
    def spacegroup(self) -> SpaceGroup: ...

    @spacegroup.setter
    def spacegroup(self, arg: SpaceGroup | None) -> None: ...

    @property
    def spacegroup_name(self) -> str: ...

    @property
    def spacegroup_number(self) -> int: ...

    @property
    def datasets(self) -> MtzDatasets: ...

    @datasets.setter
    def datasets(self, arg: MtzDatasets, /) -> None: ...

    @property
    def columns(self) -> MtzColumns: ...

    @columns.setter
    def columns(self, arg: MtzColumns, /) -> None: ...

    @property
    def batches(self) -> MtzBatches: ...

    @batches.setter
    def batches(self, arg: MtzBatches, /) -> None: ...

    @property
    def history(self) -> list[str]: ...

    @history.setter
    def history(self, arg: Sequence[str], /) -> None: ...

    @property
    def appended_text(self) -> str: ...

    @appended_text.setter
    def appended_text(self, arg: str, /) -> None: ...

    def set_logging(self, arg: object | None) -> None: ...

    def resolution_high(self) -> float: ...

    def resolution_low(self) -> float: ...

    def dataset(self, id: int) -> Mtz.Dataset: ...

    def count(self, label: str) -> int: ...

    def column_with_label(self, label: str, dataset: Mtz.Dataset | None = None, type: str = '*') -> Mtz.Column: ...

    def rfree_column(self) -> Mtz.Column: ...

    def columns_with_type(self, type: str) -> list[Mtz.Column]: ...

    def column_labels(self) -> list[str]: ...

    def get_cell(self, dataset: int = -1) -> UnitCell: ...

    def set_cell_for_all(self, arg: UnitCell, /) -> None: ...

    def make_miller_array(self) -> NDArray[numpy.int32]: ...

    def make_1_d2_array(self, dataset: int = -1) -> NDArray[numpy.float32]: ...

    def make_d_array(self, dataset: int = -1) -> NDArray[numpy.float32]: ...

    def get_size_for_hkl(self, min_size: Sequence[int] = [0, 0, 0], sample_rate: float = 0.0) -> list[int]: ...

    def data_fits_into(self, size: Sequence[int]) -> bool: ...

    def get_f_phi_on_grid(self, f: str, phi: str, size: Sequence[int], half_l: bool = False, order: AxisOrder = AxisOrder.XYZ) -> ReciprocalComplexGrid: ...

    def get_value_on_grid(self, label: str, size: Sequence[int], half_l: bool = False, order: AxisOrder = AxisOrder.XYZ) -> ReciprocalFloatGrid: ...

    def transform_f_phi_to_map(self, f: str, phi: str, min_size: Sequence[int] = [0, 0, 0], exact_size: Sequence[int] = [0, 0, 0], sample_rate: float = 0.0, order: AxisOrder = AxisOrder.XYZ) -> FloatGrid: ...

    def get_float(self, col: str, as_is: bool = False) -> FloatAsuData: ...

    def get_int(self, col: str, as_is: bool = False) -> IntAsuData: ...

    def get_f_phi(self, f: str, phi: str, as_is: bool = False) -> ComplexAsuData: ...

    def get_value_sigma(self, f: str, sigma: str, as_is: bool = False) -> ValueSigmaAsuData: ...

    def add_dataset(self, name: str) -> Mtz.Dataset: ...

    def add_column(self, label: str, type: str, dataset_id: int = -1, pos: int = -1, expand_data: bool = True) -> Mtz.Column: ...

    def replace_column(self, dest_idx: int, src_col: Mtz.Column, trailing_cols: Sequence[str] = []) -> Mtz.Column: ...

    def copy_column(self, dest_idx: int, src_col: Mtz.Column, trailing_cols: Sequence[str] = []) -> Mtz.Column: ...

    def remove_column(self, index: int) -> None: ...

    @overload
    def set_data(self, asu_data: ComplexAsuData) -> None: ...

    @overload
    def set_data(self, asu_data: FloatAsuData) -> None: ...

    @overload
    def set_data(self, array: Annotated[NDArray[numpy.float32], dict(shape=(None, None))]) -> None: ...

    def filtered(self, arg: Annotated[NDArray[numpy.bool], dict(shape=(None,), device='cpu')], /) -> Mtz: ...

    def update_reso(self) -> None: ...

    def sort(self, use_first: int = 3) -> bool: ...

    def ensure_asu(self, tnt_asu: bool = False) -> None: ...

    def switch_to_original_hkl(self) -> bool: ...

    def switch_to_asu_hkl(self) -> bool: ...

    def write_to_file(self, path: str) -> None: ...

    def write_to_bytes(self) -> bytes: ...

    def reindex(self, op: Op) -> None: ...

    def expand_to_p1(self) -> None: ...

    def row_as_dict(self, hkl: Sequence[int]) -> dict: ...

    def __repr__(self) -> str: ...

class MtzDatasets:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: MtzDatasets) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Mtz.Dataset], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Mtz.Dataset]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Mtz.Dataset: ...

    @overload
    def __getitem__(self, arg: slice, /) -> MtzDatasets: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Mtz.Dataset, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Mtz.Dataset, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Mtz.Dataset:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: MtzDatasets, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Mtz.Dataset, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: MtzDatasets, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class MtzColumns:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: MtzColumns) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Mtz.Column], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Mtz.Column]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Mtz.Column: ...

    @overload
    def __getitem__(self, arg: slice, /) -> MtzColumns: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Mtz.Column, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Mtz.Column, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Mtz.Column:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: MtzColumns, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Mtz.Column, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: MtzColumns, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class MtzBatches:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: MtzBatches) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Mtz.Batch], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Mtz.Batch]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Mtz.Batch: ...

    @overload
    def __getitem__(self, arg: slice, /) -> MtzBatches: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Mtz.Batch, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Mtz.Batch, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Mtz.Batch:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: MtzBatches, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Mtz.Batch, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: MtzBatches, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class BatchInts:
    def __getitem__(self, arg: int, /) -> int: ...

    def __setitem__(self, arg0: int, arg1: int, /) -> None: ...

    def __len__(self) -> int: ...

class BatchFloats:
    def __getitem__(self, arg: int, /) -> float: ...

    def __setitem__(self, arg0: int, arg1: float, /) -> None: ...

    def __len__(self) -> int: ...

def read_mtz_file(path: str, logging: object | None = None, with_data: bool = True) -> Mtz: ...

class ReflnBlock:
    @property
    def block(self) -> cif.Block: ...

    @property
    def entry_id(self) -> str: ...

    @property
    def cell(self) -> UnitCell: ...

    @property
    def spacegroup(self) -> SpaceGroup: ...

    @property
    def wavelength(self) -> float: ...

    @property
    def default_loop(self) -> cif.Loop: ...

    def column_labels(self) -> list[str]: ...

    def make_int_array(self, tag: str, null: int) -> Annotated[NDArray[numpy.int32], dict(shape=(None,))]: ...

    def make_float_array(self, tag: str, null: float = float('nan')) -> Annotated[NDArray[numpy.float64], dict(shape=(None,))]: ...

    def make_miller_array(self) -> Annotated[NDArray[numpy.int32], dict(shape=(None, 3))]: ...

    def make_1_d2_array(self) -> Annotated[NDArray[numpy.float64], dict(shape=(None,))]: ...

    def make_d_array(self) -> Annotated[NDArray[numpy.float64], dict(shape=(None,))]: ...

    def get_size_for_hkl(self, min_size: Sequence[int] = [0, 0, 0], sample_rate: float = 0.0) -> list[int]: ...

    def data_fits_into(self, size: Sequence[int]) -> bool: ...

    def get_f_phi_on_grid(self, f: str, phi: str, size: Sequence[int], half_l: bool = False, order: AxisOrder = AxisOrder.XYZ) -> ReciprocalComplexGrid: ...

    def get_value_on_grid(self, column: str, size: Sequence[int], half_l: bool = False, order: AxisOrder = AxisOrder.XYZ) -> ReciprocalFloatGrid: ...

    def transform_f_phi_to_map(self, f: str, phi: str, min_size: Sequence[int] = [0, 0, 0], exact_size: Sequence[int] = [0, 0, 0], sample_rate: float = 0.0, order: AxisOrder = AxisOrder.XYZ) -> FloatGrid: ...

    def get_float(self, col: str, as_is: bool = False) -> FloatAsuData: ...

    def get_int(self, col: str, as_is: bool = False) -> IntAsuData: ...

    def get_f_phi(self, f: str, phi: str, as_is: bool = False) -> ComplexAsuData: ...

    def get_value_sigma(self, f: str, sigma: str, as_is: bool = False) -> ValueSigmaAsuData: ...

    def is_merged(self) -> bool: ...

    def is_unmerged(self) -> bool: ...

    def use_unmerged(self, arg: bool, /) -> None: ...

    def __bool__(self) -> bool: ...

    def __repr__(self) -> str: ...

class ReflnBlocks:
    def __init__(self) -> None:
        """Default constructor"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[ReflnBlock]: ...

    def __getitem__(self, arg: int, /) -> ReflnBlock: ...

    def clear(self) -> None:
        """Remove all items from list."""

def as_refln_blocks(arg: cif.Document, /) -> ReflnBlocks: ...

def hkl_cif_as_refln_block(block: cif.Block) -> ReflnBlock: ...

def transform_f_phi_grid_to_map(grid: ReciprocalComplexGrid) -> FloatGrid: ...

def transform_map_to_f_phi(map: FloatGrid, half_l: bool = False, use_scale: bool = True) -> ReciprocalComplexGrid: ...

def cromer_liberman(z: int, energy: float) -> tuple[float, float]: ...

def count_reflections(cell: UnitCell, spacegroup: SpaceGroup, dmin: float, dmax: float = 0.0, unique: bool = True) -> int: ...

def make_miller_array(cell: UnitCell, spacegroup: SpaceGroup, dmin: float, dmax: float = 0.0, unique: bool = True) -> Annotated[NDArray[numpy.int32], dict(shape=(None, 3))]: ...

class CifToMtz:
    def __init__(self) -> None: ...

    @property
    def title(self) -> str: ...

    @title.setter
    def title(self, arg: str, /) -> None: ...

    @property
    def history(self) -> list[str]: ...

    @history.setter
    def history(self, arg: Sequence[str], /) -> None: ...

    @property
    def spec_lines(self) -> list[str]: ...

    @spec_lines.setter
    def spec_lines(self, arg: Sequence[str], /) -> None: ...

    def convert_block_to_mtz(self, rblock: ReflnBlock, logger: object = (None, 6)) -> Mtz: ...

class MtzToCif:
    def __init__(self) -> None: ...

    @property
    def spec_lines(self) -> list[str]: ...

    @spec_lines.setter
    def spec_lines(self, arg: Sequence[str], /) -> None: ...

    @property
    def with_comments(self) -> bool: ...

    @with_comments.setter
    def with_comments(self, arg: bool, /) -> None: ...

    @property
    def with_history(self) -> bool: ...

    @with_history.setter
    def with_history(self, arg: bool, /) -> None: ...

    @property
    def skip_empty(self) -> bool: ...

    @skip_empty.setter
    def skip_empty(self, arg: bool, /) -> None: ...

    @property
    def skip_negative_sigi(self) -> bool: ...

    @skip_negative_sigi.setter
    def skip_negative_sigi(self, arg: bool, /) -> None: ...

    @property
    def wavelength(self) -> float: ...

    @wavelength.setter
    def wavelength(self, arg: float, /) -> None: ...

    @property
    def free_flag_value(self) -> int: ...

    @free_flag_value.setter
    def free_flag_value(self, arg: int, /) -> None: ...

    def write_cif_to_string(self, mtz: Mtz, mtz2: Mtz | None = None) -> str: ...

def validate_merged_intensities(arg0: Intensities, arg1: Intensities, relaxed_check: bool = False, logger: object | None = None) -> bool: ...

def get_software_from_mtz_history(arg: Sequence[str], /) -> list[SoftwareItem]: ...

class DataType(enum.Enum):
    Unknown = 0

    Unmerged = 1

    Mean = 2

    Anomalous = 3

    MergedMA = 4

    MergedAM = 5

    UAM = 6

@overload
def check_data_type_under_symmetry(arg: ReflnBlock, /) -> tuple[DataType, int]: ...

@overload
def check_data_type_under_symmetry(arg: Mtz, /) -> tuple[DataType, int]: ...

class MergingStats:
    @property
    def all_refl(self) -> int: ...

    @property
    def unique_refl(self) -> int: ...

    @property
    def stats_refl(self) -> int: ...

    def r_merge(self) -> float: ...

    def r_meas(self) -> float: ...

    def r_pim(self) -> float: ...

    def cc_half(self) -> float: ...

    def cc_star(self) -> float: ...

class Intensities:
    def __init__(self) -> None: ...

    def clone(self) -> Intensities: ...

    def __len__(self) -> int: ...

    @property
    def spacegroup(self) -> SpaceGroup: ...

    @spacegroup.setter
    def spacegroup(self, arg: SpaceGroup, /) -> None: ...

    @property
    def unit_cell(self) -> UnitCell: ...

    @unit_cell.setter
    def unit_cell(self, arg: UnitCell, /) -> None: ...

    @property
    def type(self) -> DataType: ...

    @type.setter
    def type(self, arg: DataType, /) -> None: ...

    def resolution_range(self) -> list[float]: ...

    def remove_systematic_absences(self) -> None: ...

    def sort(self) -> None: ...

    def merge_in_place(self, new_type: DataType) -> None: ...

    def calculate_merging_stats(self, binner: Binner | None, use_weights: str = 'Y') -> list[MergingStats]: ...

    def prepare_for_merging(self, arg: DataType, /) -> DataType: ...

    def calculate_correlation(self, arg: Intensities, /) -> Correlation: ...

    def import_mtz(self, arg0: Mtz, type: DataType = DataType.Unknown) -> None: ...

    def import_xds(self, arg: XdsAscii, /) -> None: ...

    def import_refln_block(self, arg0: ReflnBlock, type: DataType = DataType.Unknown) -> None: ...

    def prepare_merged_mtz(self, with_nobs: bool) -> Mtz: ...

    @property
    def miller_array(self) -> Annotated[NDArray[numpy.int32], dict(shape=(None, 3))]: ...

    @property
    def value_array(self) -> Annotated[NDArray[numpy.float64], dict(shape=(None,))]: ...

    @property
    def sigma_array(self) -> Annotated[NDArray[numpy.float64], dict(shape=(None,))]: ...

    @property
    def nobs_array(self) -> Annotated[NDArray[numpy.int16], dict(shape=(None,))]: ...

    @property
    def isign_array(self) -> Annotated[NDArray[numpy.int8], dict(shape=(None,))]: ...

    def set_data(self, cell: UnitCell, sg: SpaceGroup, miller_array: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), device='cpu')], value_array: Annotated[NDArray[numpy.float64], dict(shape=(None,), device='cpu')], sigma_array: Annotated[NDArray[numpy.float64], dict(shape=(None,), device='cpu')]) -> None: ...

class Binner:
    def __init__(self) -> None: ...

    class Method(enum.Enum):
        EqualCount = 0

        Dstar = 1

        Dstar2 = 2

        Dstar3 = 3

    @overload
    def setup(self, nbins: int, method: Binner.Method, mtz: Mtz, cell: UnitCell | None = None) -> None: ...

    @overload
    def setup(self, nbins: int, method: Binner.Method, r: ReflnBlock, cell: UnitCell | None = None) -> None: ...

    @overload
    def setup(self, nbins: int, method: Binner.Method, intensities: Intensities) -> None: ...

    @overload
    def setup(self, nbins: int, method: Binner.Method, hkl: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), device='cpu')], cell: UnitCell) -> None: ...

    def setup_from_1_d2(self, nbins: int, method: Binner.Method, inv_d2: Annotated[NDArray[numpy.float64], dict(shape=(None,), order='C', device='cpu')], cell: UnitCell) -> None: ...

    def get_bin(self, arg: Sequence[int], /) -> int: ...

    @overload
    def get_bins(self, arg: Mtz, /) -> Annotated[NDArray[numpy.int32], dict(shape=(None,))]: ...

    @overload
    def get_bins(self, arg: ReflnBlock, /) -> Annotated[NDArray[numpy.int32], dict(shape=(None,))]: ...

    @overload
    def get_bins(self, arg: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), device='cpu')], /) -> Annotated[NDArray[numpy.int32], dict(shape=(None,))]: ...

    def get_bins_from_1_d2(self, arg: Annotated[NDArray[numpy.float64], dict(shape=(None,), order='C', device='cpu')], /) -> Annotated[NDArray[numpy.int32], dict(shape=(None,))]: ...

    def dmin_of_bin(self, arg: int, /) -> float: ...

    def dmax_of_bin(self, arg: int, /) -> float: ...

    @property
    def size(self) -> int: ...

    @property
    def limits(self) -> list[float]: ...

    @property
    def cell(self) -> UnitCell: ...

    @cell.setter
    def cell(self, arg: UnitCell, /) -> None: ...

    @property
    def min_1_d2(self) -> float: ...

    @property
    def max_1_d2(self) -> float: ...

def combine_correlations(arg: Sequence[Correlation], /) -> Correlation: ...

def calculate_amplitude_normalizers(arg0: Mtz, arg1: str, arg2: Binner, /) -> Annotated[NDArray[numpy.float64], dict(shape=(None,))]: ...

class HklMatch:
    def __init__(self, hkl: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), order='C', device='cpu')], ref: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), order='C', device='cpu')]) -> None: ...

    def aligned(self, arg: Annotated[NDArray[numpy.float64], dict(shape=(None,), order='C', device='cpu')], /) -> Annotated[NDArray[numpy.float64], dict(shape=(None,))]: ...

    @property
    def pos(self) -> list[int]: ...

class ChemComp:
    def __init__(self) -> None: ...

    class Atom:
        def __init__(self) -> None: ...

        @property
        def id(self) -> str: ...

        @id.setter
        def id(self, arg: str, /) -> None: ...

        @property
        def el(self) -> Element: ...

        @el.setter
        def el(self, arg: Element, /) -> None: ...

        @property
        def xyz(self) -> Position: ...

        @xyz.setter
        def xyz(self, arg: Position, /) -> None: ...

        @property
        def charge(self) -> float: ...

        @charge.setter
        def charge(self, arg: float, /) -> None: ...

        @property
        def chem_type(self) -> str: ...

        @chem_type.setter
        def chem_type(self, arg: str, /) -> None: ...

        def is_hydrogen(self) -> bool: ...

    class Group(enum.Enum):
        Peptide = 0

        PPeptide = 1

        MPeptide = 2

        Dna = 3

        Rna = 4

        DnaRna = 5

        Pyranose = 6

        Ketopyranose = 7

        Furanose = 8

        NonPolymer = 9

        Null = 10

    class Aliasing:
        @property
        def group(self) -> ChemComp.Group: ...

        def name_from_alias(self, arg: str, /) -> str: ...

    @property
    def name(self) -> str: ...

    @name.setter
    def name(self, arg: str, /) -> None: ...

    @property
    def group(self) -> ChemComp.Group: ...

    @group.setter
    def group(self, arg: ChemComp.Group, /) -> None: ...

    @property
    def atoms(self) -> ChemCompAtoms: ...

    @property
    def rt(self) -> Restraints: ...

    @staticmethod
    def read_group(arg: str, /) -> ChemComp.Group: ...

    @staticmethod
    def group_str(arg: ChemComp.Group, /) -> str: ...

    def set_group(self, arg: str, /) -> None: ...

    def get_atom(self, arg: str, /) -> ChemComp.Atom: ...

    def find_atom(self, arg: str, /) -> ChemComp.Atom: ...

    def remove_hydrogens(self) -> ChemComp: ...

class Restraints:
    def __init__(self) -> None: ...

    class Bond:
        def __init__(self) -> None: ...

        @property
        def id1(self) -> Restraints.AtomId: ...

        @id1.setter
        def id1(self, arg: Restraints.AtomId, /) -> None: ...

        @property
        def id2(self) -> Restraints.AtomId: ...

        @id2.setter
        def id2(self, arg: Restraints.AtomId, /) -> None: ...

        @property
        def type(self) -> BondType: ...

        @type.setter
        def type(self, arg: BondType, /) -> None: ...

        @property
        def aromatic(self) -> bool: ...

        @aromatic.setter
        def aromatic(self, arg: bool, /) -> None: ...

        @property
        def value(self) -> float: ...

        @value.setter
        def value(self, arg: float, /) -> None: ...

        @property
        def esd(self) -> float: ...

        @esd.setter
        def esd(self, arg: float, /) -> None: ...

        @property
        def value_nucleus(self) -> float: ...

        @value_nucleus.setter
        def value_nucleus(self, arg: float, /) -> None: ...

        @property
        def esd_nucleus(self) -> float: ...

        @esd_nucleus.setter
        def esd_nucleus(self, arg: float, /) -> None: ...

        def lexicographic_str(self) -> str: ...

        def __repr__(self) -> str: ...

    class Angle:
        def __init__(self) -> None: ...

        @property
        def id1(self) -> Restraints.AtomId: ...

        @id1.setter
        def id1(self, arg: Restraints.AtomId, /) -> None: ...

        @property
        def id2(self) -> Restraints.AtomId: ...

        @id2.setter
        def id2(self, arg: Restraints.AtomId, /) -> None: ...

        @property
        def id3(self) -> Restraints.AtomId: ...

        @id3.setter
        def id3(self, arg: Restraints.AtomId, /) -> None: ...

        @property
        def value(self) -> float: ...

        @value.setter
        def value(self, arg: float, /) -> None: ...

        @property
        def esd(self) -> float: ...

        @esd.setter
        def esd(self, arg: float, /) -> None: ...

        def __repr__(self) -> str: ...

    class Torsion:
        def __init__(self) -> None: ...

        @property
        def label(self) -> str: ...

        @label.setter
        def label(self, arg: str, /) -> None: ...

        @property
        def id1(self) -> Restraints.AtomId: ...

        @id1.setter
        def id1(self, arg: Restraints.AtomId, /) -> None: ...

        @property
        def id2(self) -> Restraints.AtomId: ...

        @id2.setter
        def id2(self, arg: Restraints.AtomId, /) -> None: ...

        @property
        def id3(self) -> Restraints.AtomId: ...

        @id3.setter
        def id3(self, arg: Restraints.AtomId, /) -> None: ...

        @property
        def id4(self) -> Restraints.AtomId: ...

        @id4.setter
        def id4(self, arg: Restraints.AtomId, /) -> None: ...

        @property
        def value(self) -> float: ...

        @value.setter
        def value(self, arg: float, /) -> None: ...

        @property
        def esd(self) -> float: ...

        @esd.setter
        def esd(self, arg: float, /) -> None: ...

        @property
        def period(self) -> int: ...

        @period.setter
        def period(self, arg: int, /) -> None: ...

        def __repr__(self) -> str: ...

    class Chirality:
        def __init__(self) -> None: ...

        @property
        def id_ctr(self) -> Restraints.AtomId: ...

        @id_ctr.setter
        def id_ctr(self, arg: Restraints.AtomId, /) -> None: ...

        @property
        def id1(self) -> Restraints.AtomId: ...

        @id1.setter
        def id1(self, arg: Restraints.AtomId, /) -> None: ...

        @property
        def id2(self) -> Restraints.AtomId: ...

        @id2.setter
        def id2(self, arg: Restraints.AtomId, /) -> None: ...

        @property
        def id3(self) -> Restraints.AtomId: ...

        @id3.setter
        def id3(self, arg: Restraints.AtomId, /) -> None: ...

        @property
        def sign(self) -> ChiralityType: ...

        @sign.setter
        def sign(self, arg: ChiralityType, /) -> None: ...

        def is_wrong(self, arg: float, /) -> bool: ...

        def __repr__(self) -> str: ...

    class Plane:
        def __init__(self) -> None: ...

        @property
        def label(self) -> str: ...

        @label.setter
        def label(self, arg: str, /) -> None: ...

        @property
        def ids(self) -> list[Restraints.AtomId]: ...

        @ids.setter
        def ids(self, arg: Sequence[Restraints.AtomId], /) -> None: ...

        @property
        def esd(self) -> float: ...

        @esd.setter
        def esd(self, arg: float, /) -> None: ...

        def __repr__(self) -> str: ...

    class DistanceOf(enum.Enum):
        ElectronCloud = 0

        Nucleus = 1

    class AtomId:
        @overload
        def __init__(self, arg0: int, arg1: str, /) -> None: ...

        @overload
        def __init__(self, arg: str, /) -> None: ...

        @property
        def comp(self) -> int: ...

        @comp.setter
        def comp(self, arg: int, /) -> None: ...

        @property
        def atom(self) -> str: ...

        @atom.setter
        def atom(self, arg: str, /) -> None: ...

        def get_from(self, res1: Residue, res2: Residue, altloc1: str, altloc2: str) -> Atom: ...

        def __repr__(self) -> str: ...

    @property
    def bonds(self) -> RestraintsBonds: ...

    @bonds.setter
    def bonds(self, arg: RestraintsBonds, /) -> None: ...

    @property
    def angles(self) -> RestraintsAngles: ...

    @angles.setter
    def angles(self, arg: RestraintsAngles, /) -> None: ...

    @property
    def torsions(self) -> RestraintsTorsions: ...

    @torsions.setter
    def torsions(self, arg: RestraintsTorsions, /) -> None: ...

    @property
    def chirs(self) -> RestraintsChirs: ...

    @chirs.setter
    def chirs(self, arg: RestraintsChirs, /) -> None: ...

    @property
    def planes(self) -> RestraintsPlanes: ...

    @planes.setter
    def planes(self, arg: RestraintsPlanes, /) -> None: ...

    def empty(self) -> bool: ...

    @overload
    def get_bond(self, arg0: Restraints.AtomId, arg1: Restraints.AtomId, /) -> Restraints.Bond: ...

    @overload
    def get_bond(self, arg0: str, arg1: str, /) -> Restraints.Bond: ...

    def find_shortest_path(self, arg0: Restraints.AtomId, arg1: Restraints.AtomId, arg2: Sequence[Restraints.AtomId], /) -> list[Restraints.AtomId]: ...

    def chiral_abs_volume(self, arg: Restraints.Chirality, /) -> float: ...

class RestraintsBonds:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: RestraintsBonds) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Restraints.Bond], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Restraints.Bond]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Restraints.Bond: ...

    @overload
    def __getitem__(self, arg: slice, /) -> RestraintsBonds: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Restraints.Bond, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Restraints.Bond, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Restraints.Bond:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: RestraintsBonds, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Restraints.Bond, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: RestraintsBonds, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class RestraintsAngles:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: RestraintsAngles) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Restraints.Angle], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Restraints.Angle]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Restraints.Angle: ...

    @overload
    def __getitem__(self, arg: slice, /) -> RestraintsAngles: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Restraints.Angle, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Restraints.Angle, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Restraints.Angle:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: RestraintsAngles, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Restraints.Angle, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: RestraintsAngles, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class RestraintsTorsions:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: RestraintsTorsions) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Restraints.Torsion], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Restraints.Torsion]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Restraints.Torsion: ...

    @overload
    def __getitem__(self, arg: slice, /) -> RestraintsTorsions: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Restraints.Torsion, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Restraints.Torsion, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Restraints.Torsion:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: RestraintsTorsions, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Restraints.Torsion, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: RestraintsTorsions, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class RestraintsChirs:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: RestraintsChirs) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Restraints.Chirality], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Restraints.Chirality]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Restraints.Chirality: ...

    @overload
    def __getitem__(self, arg: slice, /) -> RestraintsChirs: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Restraints.Chirality, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Restraints.Chirality, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Restraints.Chirality:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: RestraintsChirs, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Restraints.Chirality, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: RestraintsChirs, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class RestraintsPlanes:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: RestraintsPlanes) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Restraints.Plane], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Restraints.Plane]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Restraints.Plane: ...

    @overload
    def __getitem__(self, arg: slice, /) -> RestraintsPlanes: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Restraints.Plane, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Restraints.Plane, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Restraints.Plane:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: RestraintsPlanes, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Restraints.Plane, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: RestraintsPlanes, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class ChemCompAtoms:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: ChemCompAtoms) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[ChemComp.Atom], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[ChemComp.Atom]: ...

    @overload
    def __getitem__(self, arg: int, /) -> ChemComp.Atom: ...

    @overload
    def __getitem__(self, arg: slice, /) -> ChemCompAtoms: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: ChemComp.Atom, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: ChemComp.Atom, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> ChemComp.Atom:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: ChemCompAtoms, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: ChemComp.Atom, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: ChemCompAtoms, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class BondType(enum.Enum):
    Unspec = 0

    Single = 1

    Double = 2

    Triple = 3

    Aromatic = 4

    Deloc = 5

    Metal = 6

class ChiralityType(enum.Enum):
    Positive = 0

    Negative = 1

    Both = 2

def make_chemcomp_from_block(arg: cif.Block, /) -> ChemComp: ...

def add_chemcomp_to_block(arg0: ChemComp, arg1: cif.Block, /) -> None: ...

class ChemMod:
    def __init__(self) -> None: ...

    class AtomMod:
        @property
        def func(self) -> int: ...

        @func.setter
        def func(self, arg: int, /) -> None: ...

        @property
        def old_id(self) -> str: ...

        @old_id.setter
        def old_id(self, arg: str, /) -> None: ...

        @property
        def new_id(self) -> str: ...

        @new_id.setter
        def new_id(self, arg: str, /) -> None: ...

        @property
        def el(self) -> Element: ...

        @el.setter
        def el(self, arg: Element, /) -> None: ...

        @property
        def charge(self) -> float: ...

        @charge.setter
        def charge(self, arg: float, /) -> None: ...

        @property
        def chem_type(self) -> str: ...

        @chem_type.setter
        def chem_type(self, arg: str, /) -> None: ...

    @property
    def id(self) -> str: ...

    @id.setter
    def id(self, arg: str, /) -> None: ...

    @property
    def name(self) -> str: ...

    @name.setter
    def name(self, arg: str, /) -> None: ...

    @property
    def comp_id(self) -> str: ...

    @comp_id.setter
    def comp_id(self, arg: str, /) -> None: ...

    @property
    def group_id(self) -> str: ...

    @group_id.setter
    def group_id(self, arg: str, /) -> None: ...

    @property
    def atom_mods(self) -> list[ChemMod.AtomMod]: ...

    @atom_mods.setter
    def atom_mods(self, arg: Sequence[ChemMod.AtomMod], /) -> None: ...

    @property
    def rt(self) -> Restraints: ...

    @rt.setter
    def rt(self, arg: Restraints, /) -> None: ...

    def __repr__(self) -> str: ...

class ChemLink:
    def __init__(self) -> None: ...

    class Side:
        def __init__(self) -> None: ...

        @property
        def comp(self) -> str: ...

        @comp.setter
        def comp(self, arg: str, /) -> None: ...

        @property
        def mod(self) -> str: ...

        @mod.setter
        def mod(self, arg: str, /) -> None: ...

        @property
        def group(self) -> ChemComp.Group: ...

        @group.setter
        def group(self, arg: ChemComp.Group, /) -> None: ...

        def __repr__(self) -> str: ...

    @property
    def id(self) -> str: ...

    @id.setter
    def id(self, arg: str, /) -> None: ...

    @property
    def name(self) -> str: ...

    @name.setter
    def name(self, arg: str, /) -> None: ...

    @property
    def side1(self) -> ChemLink.Side: ...

    @side1.setter
    def side1(self, arg: ChemLink.Side, /) -> None: ...

    @property
    def side2(self) -> ChemLink.Side: ...

    @side2.setter
    def side2(self, arg: ChemLink.Side, /) -> None: ...

    @property
    def rt(self) -> Restraints: ...

    @rt.setter
    def rt(self, arg: Restraints, /) -> None: ...

    def __repr__(self) -> str: ...

class ChemCompMap:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: ChemCompMap) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: dict[str, ChemComp], /) -> None:
        """Construct from a dictionary"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the map is nonempty"""

    def __repr__(self) -> str: ...

    @overload
    def __contains__(self, arg: str, /) -> bool: ...

    @overload
    def __contains__(self, arg: object, /) -> bool: ...

    def __iter__(self) -> Iterator[str]: ...

    def __getitem__(self, arg: str, /) -> ChemComp: ...

    def __delitem__(self, arg: str, /) -> None: ...

    def clear(self) -> None:
        """Remove all items"""

    def __setitem__(self, arg0: str, arg1: ChemComp, /) -> None: ...

    def update(self, arg: ChemCompMap, /) -> None:
        """Update the map with element from `arg`"""

    class ItemView:
        def __len__(self) -> int: ...

        def __iter__(self) -> Iterator[tuple[str, ChemComp]]: ...

    class KeyView:
        @overload
        def __contains__(self, arg: str, /) -> bool: ...

        @overload
        def __contains__(self, arg: object, /) -> bool: ...

        def __len__(self) -> int: ...

        def __iter__(self) -> Iterator[str]: ...

    class ValueView:
        def __len__(self) -> int: ...

        def __iter__(self) -> Iterator[ChemComp]: ...

    def keys(self) -> ChemCompMap.KeyView:
        """Returns an iterable view of the map's keys."""

    def values(self) -> ChemCompMap.ValueView:
        """Returns an iterable view of the map's values."""

    def items(self) -> ChemCompMap.ItemView:
        """Returns an iterable view of the map's items."""

class ChemLinkMap:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: ChemLinkMap) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: dict[str, ChemLink], /) -> None:
        """Construct from a dictionary"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the map is nonempty"""

    def __repr__(self) -> str: ...

    @overload
    def __contains__(self, arg: str, /) -> bool: ...

    @overload
    def __contains__(self, arg: object, /) -> bool: ...

    def __iter__(self) -> Iterator[str]: ...

    def __getitem__(self, arg: str, /) -> ChemLink: ...

    def __delitem__(self, arg: str, /) -> None: ...

    def clear(self) -> None:
        """Remove all items"""

    def __setitem__(self, arg0: str, arg1: ChemLink, /) -> None: ...

    def update(self, arg: ChemLinkMap, /) -> None:
        """Update the map with element from `arg`"""

    class ItemView:
        def __len__(self) -> int: ...

        def __iter__(self) -> Iterator[tuple[str, ChemLink]]: ...

    class KeyView:
        @overload
        def __contains__(self, arg: str, /) -> bool: ...

        @overload
        def __contains__(self, arg: object, /) -> bool: ...

        def __len__(self) -> int: ...

        def __iter__(self) -> Iterator[str]: ...

    class ValueView:
        def __len__(self) -> int: ...

        def __iter__(self) -> Iterator[ChemLink]: ...

    def keys(self) -> ChemLinkMap.KeyView:
        """Returns an iterable view of the map's keys."""

    def values(self) -> ChemLinkMap.ValueView:
        """Returns an iterable view of the map's values."""

    def items(self) -> ChemLinkMap.ItemView:
        """Returns an iterable view of the map's items."""

class ChemModMap:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: ChemModMap) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: dict[str, ChemMod], /) -> None:
        """Construct from a dictionary"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the map is nonempty"""

    def __repr__(self) -> str: ...

    @overload
    def __contains__(self, arg: str, /) -> bool: ...

    @overload
    def __contains__(self, arg: object, /) -> bool: ...

    def __iter__(self) -> Iterator[str]: ...

    def __getitem__(self, arg: str, /) -> ChemMod: ...

    def __delitem__(self, arg: str, /) -> None: ...

    def clear(self) -> None:
        """Remove all items"""

    def __setitem__(self, arg0: str, arg1: ChemMod, /) -> None: ...

    def update(self, arg: ChemModMap, /) -> None:
        """Update the map with element from `arg`"""

    class ItemView:
        def __len__(self) -> int: ...

        def __iter__(self) -> Iterator[tuple[str, ChemMod]]: ...

    class KeyView:
        @overload
        def __contains__(self, arg: str, /) -> bool: ...

        @overload
        def __contains__(self, arg: object, /) -> bool: ...

        def __len__(self) -> int: ...

        def __iter__(self) -> Iterator[str]: ...

    class ValueView:
        def __len__(self) -> int: ...

        def __iter__(self) -> Iterator[ChemMod]: ...

    def keys(self) -> ChemModMap.KeyView:
        """Returns an iterable view of the map's keys."""

    def values(self) -> ChemModMap.ValueView:
        """Returns an iterable view of the map's values."""

    def items(self) -> ChemModMap.ItemView:
        """Returns an iterable view of the map's items."""

class EnerLib:
    pass

class MonLib:
    def __init__(self) -> None: ...

    @property
    def monomer_dir(self) -> str: ...

    @monomer_dir.setter
    def monomer_dir(self, arg: str, /) -> None: ...

    @property
    def ener_lib(self) -> EnerLib: ...

    @property
    def monomers(self) -> ChemCompMap: ...

    @property
    def links(self) -> ChemLinkMap: ...

    @property
    def modifications(self) -> ChemModMap: ...

    def get_link(self, link_id: str) -> ChemLink: ...

    def get_mod(self, name: str) -> ChemMod: ...

    def match_link(self, res1: Residue, atom1: str, alt1: str, res2: Residue, atom2: str, alt2: str, min_bond_sq: float = 0.0) -> tuple[ChemLink, bool, ChemComp.Aliasing, ChemComp.Aliasing]: ...

    def test_link(self, link: ChemLink, res1: str, atom1: str, res2: str, atom2: str) -> tuple[bool, ChemComp.Aliasing, ChemComp.Aliasing]: ...

    def add_monomer_if_present(self, arg: cif.Block, /) -> None: ...

    def read_monomer_doc(self, arg: cif.Document, /) -> None: ...

    def read_monomer_cif(self, arg: str, /) -> None: ...

    def read_monomer_lib(self, monomer_dir: str, resnames: Sequence[str], logging: object | None = None) -> bool: ...

    def find_ideal_distance(self, arg0: CRA, arg1: CRA, /) -> float: ...

    def update_old_atom_names(self, st: Structure, logging: object | None = None) -> None: ...

    def path(self, code: str = '') -> str: ...

    def __repr__(self) -> str: ...

    def clone(self) -> MonLib: ...

def read_monomer_lib(monomer_dir: str, resnames: Sequence[str], libin: str = '', ignore_missing: bool = False) -> MonLib: ...

class Topo:
    def __init__(self) -> None: ...

    class Bond:
        @property
        def restr(self) -> Restraints.Bond: ...

        @property
        def atoms(self) -> list[Atom]: ...

        def calculate(self) -> float: ...

        def calculate_z(self) -> float: ...

    class Angle:
        @property
        def restr(self) -> Restraints.Angle: ...

        @property
        def atoms(self) -> list[Atom]: ...

        def calculate(self) -> float: ...

        def calculate_z(self) -> float: ...

    class Torsion:
        @property
        def restr(self) -> Restraints.Torsion: ...

        @property
        def atoms(self) -> list[Atom]: ...

        def calculate(self) -> float: ...

        def calculate_z(self) -> float: ...

    class Chirality:
        @property
        def restr(self) -> Restraints.Chirality: ...

        @property
        def atoms(self) -> list[Atom]: ...

        def calculate(self) -> float: ...

        def calculate_z(self, ideal_abs_vol: float, esd: float) -> float: ...

        def check(self) -> bool: ...

    class Plane:
        @property
        def restr(self) -> Restraints.Plane: ...

        @property
        def atoms(self) -> list[Atom]: ...

        def has(self, arg: Atom, /) -> bool: ...

    class Rule:
        @property
        def rkind(self) -> RKind: ...

        @property
        def index(self) -> int: ...

    class Mod:
        @property
        def id(self) -> str: ...

        @property
        def alias(self) -> ChemComp.Group: ...

        @property
        def altloc(self) -> str: ...

    class FinalChemComp:
        @property
        def altloc(self) -> str: ...

        @property
        def cc(self) -> ChemComp: ...

    class Link:
        @property
        def link_id(self) -> str: ...

        @property
        def res1(self) -> Residue: ...

        @property
        def res2(self) -> Residue: ...

        @property
        def alt1(self) -> str: ...

        @property
        def alt2(self) -> str: ...

        @property
        def link_rules(self) -> TopoRules: ...

    class ResInfo:
        @property
        def res(self) -> Residue: ...

        @property
        def prev(self) -> TopoLinks: ...

        @property
        def mods(self) -> TopoMods: ...

        @property
        def chemcomps(self) -> TopoFinalChemComps: ...

        @property
        def monomer_rules(self) -> TopoRules: ...

        def get_final_chemcomp(self, arg: str, /) -> ChemComp: ...

    class ChainInfo:
        @property
        def chain_ref(self) -> Chain: ...

        @property
        def subchain_name(self) -> str: ...

        @property
        def entity_id(self) -> str: ...

        @property
        def polymer(self) -> bool: ...

        @property
        def polymer_type(self) -> PolymerType: ...

        @property
        def res_infos(self) -> TopoResInfos: ...

    def adjust_hydrogen_distances(self, of: Restraints.DistanceOf, default_scale: float = 1.0) -> None: ...

    @property
    def bonds(self) -> TopoBonds: ...

    @property
    def angles(self) -> TopoAngles: ...

    @property
    def torsions(self) -> TopoTorsions: ...

    @property
    def chirs(self) -> TopoChirs: ...

    @property
    def planes(self) -> TopoPlanes: ...

    @property
    def extras(self) -> TopoLinks: ...

    @property
    def chain_infos(self) -> TopoChainInfos: ...

    def ideal_chiral_abs_volume(self, arg: Topo.Chirality, /) -> float: ...

    def links_to_previous(self, arg: Residue, /) -> TopoLinks: ...

    def first_bond_in_link(self, arg: Topo.Link, /) -> Topo.Bond: ...

    def set_cispeps_in_structure(self, arg: Structure, /) -> None: ...

    def find_missing_atoms(self, including_hydrogen: bool = False) -> list[AtomAddress]: ...

class HydrogenChange(enum.Enum):
    NoChange = 0

    Shift = 1

    Remove = 2

    ReAdd = 3

    ReAddButWater = 4

    ReAddKnown = 5

class TopoBonds:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: TopoBonds) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Topo.Bond], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Topo.Bond]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Topo.Bond: ...

    @overload
    def __getitem__(self, arg: slice, /) -> TopoBonds: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Topo.Bond, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Topo.Bond, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Topo.Bond:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: TopoBonds, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Topo.Bond, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: TopoBonds, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class TopoAngles:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: TopoAngles) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Topo.Angle], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Topo.Angle]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Topo.Angle: ...

    @overload
    def __getitem__(self, arg: slice, /) -> TopoAngles: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Topo.Angle, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Topo.Angle, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Topo.Angle:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: TopoAngles, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Topo.Angle, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: TopoAngles, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class TopoTorsions:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: TopoTorsions) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Topo.Torsion], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Topo.Torsion]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Topo.Torsion: ...

    @overload
    def __getitem__(self, arg: slice, /) -> TopoTorsions: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Topo.Torsion, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Topo.Torsion, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Topo.Torsion:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: TopoTorsions, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Topo.Torsion, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: TopoTorsions, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class TopoChirs:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: TopoChirs) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Topo.Chirality], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Topo.Chirality]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Topo.Chirality: ...

    @overload
    def __getitem__(self, arg: slice, /) -> TopoChirs: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Topo.Chirality, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Topo.Chirality, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Topo.Chirality:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: TopoChirs, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Topo.Chirality, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: TopoChirs, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class TopoPlanes:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: TopoPlanes) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Topo.Plane], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Topo.Plane]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Topo.Plane: ...

    @overload
    def __getitem__(self, arg: slice, /) -> TopoPlanes: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Topo.Plane, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Topo.Plane, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Topo.Plane:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: TopoPlanes, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Topo.Plane, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: TopoPlanes, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class RKind(enum.Enum):
    Bond = 0

    Angle = 1

    Torsion = 2

    Chirality = 3

    Plane = 4

class TopoRules:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: TopoRules) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Topo.Rule], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Topo.Rule]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Topo.Rule: ...

    @overload
    def __getitem__(self, arg: slice, /) -> TopoRules: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Topo.Rule, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Topo.Rule, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Topo.Rule:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: TopoRules, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Topo.Rule, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: TopoRules, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class TopoMods:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: TopoMods) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Topo.Mod], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Topo.Mod]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Topo.Mod: ...

    @overload
    def __getitem__(self, arg: slice, /) -> TopoMods: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Topo.Mod, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Topo.Mod, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Topo.Mod:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: TopoMods, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Topo.Mod, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: TopoMods, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

    def __eq__(self, arg: object, /) -> bool: ...

    def __ne__(self, arg: object, /) -> bool: ...

    @overload
    def __contains__(self, arg: Topo.Mod, /) -> bool: ...

    @overload
    def __contains__(self, arg: object, /) -> bool: ...

    def count(self, arg: Topo.Mod, /) -> int:
        """Return number of occurrences of `arg`."""

    def remove(self, arg: Topo.Mod, /) -> None:
        """Remove first occurrence of `arg`."""

class TopoFinalChemComps:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: TopoFinalChemComps) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Topo.FinalChemComp], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Topo.FinalChemComp]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Topo.FinalChemComp: ...

    @overload
    def __getitem__(self, arg: slice, /) -> TopoFinalChemComps: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Topo.FinalChemComp, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Topo.FinalChemComp, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Topo.FinalChemComp:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: TopoFinalChemComps, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Topo.FinalChemComp, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: TopoFinalChemComps, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class TopoLinks:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: TopoLinks) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Topo.Link], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Topo.Link]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Topo.Link: ...

    @overload
    def __getitem__(self, arg: slice, /) -> TopoLinks: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Topo.Link, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Topo.Link, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Topo.Link:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: TopoLinks, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Topo.Link, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: TopoLinks, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class TopoResInfos:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: TopoResInfos) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[Topo.ResInfo], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[Topo.ResInfo]: ...

    @overload
    def __getitem__(self, arg: int, /) -> Topo.ResInfo: ...

    @overload
    def __getitem__(self, arg: slice, /) -> TopoResInfos: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: Topo.ResInfo, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: Topo.ResInfo, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> Topo.ResInfo:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: TopoResInfos, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: Topo.ResInfo, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: TopoResInfos, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

class TopoChainInfos:
    def __len__(self) -> int: ...

    def __getitem__(self, arg: int, /) -> Topo.ChainInfo: ...

def prepare_topology(st: Structure, monlib: MonLib, model_index: int = 0, h_change: HydrogenChange = HydrogenChange.NoChange, reorder: bool = False, warnings: object | None = None, ignore_unknown_links: bool = False, use_cispeps: bool = False) -> Topo: ...

def setup_for_crd(arg: Structure, /) -> None: ...

def prepare_refmac_crd(arg0: Structure, arg1: Topo, arg2: MonLib, arg3: HydrogenChange, /) -> cif.Document: ...

def add_automatic_links(arg0: Model, arg1: Structure, arg2: MonLib, /) -> None: ...

def add_dictionary_blocks(arg0: cif.Document, arg1: Sequence[str], arg2: Topo, arg3: MonLib, /) -> None: ...

class AlignmentResult:
    @property
    def score(self) -> int: ...

    @property
    def match_count(self) -> int: ...

    @property
    def match_string(self) -> str: ...

    def cigar_str(self) -> str: ...

    def calculate_identity(self, which: int = 0) -> float: ...

    def add_gaps(self, s: str, which: int) -> str: ...

    def formatted(self, arg0: str, arg1: str, /) -> str: ...

class AlignmentScoring:
    def __init__(self, what: str = 's') -> None: ...

    @property
    def match(self) -> int: ...

    @match.setter
    def match(self, arg: int, /) -> None: ...

    @property
    def mismatch(self) -> int: ...

    @mismatch.setter
    def mismatch(self, arg: int, /) -> None: ...

    @property
    def gapo(self) -> int: ...

    @gapo.setter
    def gapo(self, arg: int, /) -> None: ...

    @property
    def gape(self) -> int: ...

    @gape.setter
    def gape(self, arg: int, /) -> None: ...

    @property
    def good_gapo(self) -> int: ...

    @good_gapo.setter
    def good_gapo(self, arg: int, /) -> None: ...

    @property
    def bad_gapo(self) -> int: ...

    @bad_gapo.setter
    def bad_gapo(self, arg: int, /) -> None: ...

def align_string_sequences(query: Sequence[str], target: Sequence[str], target_gapo: Sequence[int], scoring: AlignmentScoring | None = None) -> AlignmentResult: ...

def align_sequence_to_polymer(full_seq: Sequence[str], polymer: ResidueSpan, polymer_type: PolymerType, scoring: AlignmentScoring | None = None) -> AlignmentResult: ...

class SupSelect(enum.Enum):
    CaP = 0

    MainChain = 1

    All = 2

class SupResult:
    @property
    def rmsd(self) -> float: ...

    @property
    def count(self) -> int: ...

    @property
    def center1(self) -> Position: ...

    @property
    def center2(self) -> Position: ...

    @property
    def transform(self) -> Transform: ...

def calculate_current_rmsd(fixed: ResidueSpan, movable: ResidueSpan, ptype: PolymerType, sel: SupSelect, altloc: str = '\x00') -> SupResult: ...

def calculate_superposition(fixed: ResidueSpan, movable: ResidueSpan, ptype: PolymerType, sel: SupSelect, trim_cycles: int = 0, trim_cutoff: float = 2.0, altloc: str = '\x00') -> SupResult: ...

def calculate_superpositions_in_moving_window(fixed: ResidueSpan, movable: ResidueSpan, ptype: PolymerType, radius: float = 10.0) -> list[SupResult]: ...

def superpose_positions(pos1: Sequence[Position], pos2: Sequence[Position], weight: Sequence[float] = []) -> SupResult: ...

class NeighborSearch:
    @overload
    def __init__(self, model: Model, cell: UnitCell, max_radius: float) -> None: ...

    @overload
    def __init__(self, st: Structure, max_radius: float, model_index: int = 0) -> None: ...

    @overload
    def __init__(self, small_structure: SmallStructure, max_radius: float) -> None: ...

    class Mark:
        @property
        def pos(self) -> Position: ...

        @property
        def altloc(self) -> str: ...

        @property
        def element(self) -> Element: ...

        @property
        def image_idx(self) -> int: ...

        @property
        def chain_idx(self) -> int: ...

        @property
        def residue_idx(self) -> int: ...

        @property
        def atom_idx(self) -> int: ...

        def to_cra(self, arg: Model, /) -> CRA: ...

        def to_site(self, arg: SmallStructure, /) -> SmallStructure.Site: ...

        def __repr__(self) -> str: ...

    @property
    def radius_specified(self) -> float: ...

    def populate(self, include_h: bool = True) -> NeighborSearch:
        """Usually run after constructing NeighborSearch."""

    def add_chain(self, chain: Chain, include_h: bool = True) -> None: ...

    def add_atom(self, atom: Atom, n_ch: int, n_res: int, n_atom: int) -> None:
        """Lower-level alternative to populate()"""

    def add_site(self, site: SmallStructure.Site, n: int) -> None:
        """Lower-level alternative to populate() for SmallStructure"""

    def find_atoms(self, pos: Position, alt: str = '\x00', *, min_dist: float = 0, radius: float = 0) -> VectorMarkPtr: ...

    def find_neighbors(self, atom: Atom, min_dist: float = 0, max_dist: float = 0) -> VectorMarkPtr: ...

    def find_nearest_atom(self, pos: Position, radius: float = float('inf')) -> NeighborSearch.Mark: ...

    def find_site_neighbors(self, atom: SmallStructure.Site, min_dist: float = 0, max_dist: float = 0) -> VectorMarkPtr: ...

    def dist(self, arg0: Position, arg1: Position, /) -> float: ...

    def get_image_transformation(self, arg: int, /) -> FTransform: ...

    @property
    def grid_cell(self) -> UnitCell: ...

    def __repr__(self) -> str: ...

class VectorMarkPtr:
    @overload
    def __init__(self) -> None:
        """Default constructor"""

    @overload
    def __init__(self, arg: VectorMarkPtr) -> None:
        """Copy constructor"""

    @overload
    def __init__(self, arg: Iterable[NeighborSearch.Mark], /) -> None:
        """Construct from an iterable object"""

    def __len__(self) -> int: ...

    def __bool__(self) -> bool:
        """Check whether the vector is nonempty"""

    def __repr__(self) -> str: ...

    def __iter__(self) -> Iterator[NeighborSearch.Mark]: ...

    @overload
    def __getitem__(self, arg: int, /) -> NeighborSearch.Mark: ...

    @overload
    def __getitem__(self, arg: slice, /) -> VectorMarkPtr: ...

    def clear(self) -> None:
        """Remove all items from list."""

    def append(self, arg: NeighborSearch.Mark, /) -> None:
        """Append `arg` to the end of the list."""

    def insert(self, arg0: int, arg1: NeighborSearch.Mark, /) -> None:
        """Insert object `arg1` before index `arg0`."""

    def pop(self, index: int = -1) -> NeighborSearch.Mark:
        """Remove and return item at `index` (default last)."""

    def extend(self, arg: VectorMarkPtr, /) -> None:
        """Extend `self` by appending elements from `arg`."""

    @overload
    def __setitem__(self, arg0: int, arg1: NeighborSearch.Mark, /) -> None: ...

    @overload
    def __setitem__(self, arg0: slice, arg1: VectorMarkPtr, /) -> None: ...

    @overload
    def __delitem__(self, arg: int, /) -> None: ...

    @overload
    def __delitem__(self, arg: slice, /) -> None: ...

    def __eq__(self, arg: object, /) -> bool: ...

    def __ne__(self, arg: object, /) -> bool: ...

    @overload
    def __contains__(self, arg: NeighborSearch.Mark, /) -> bool: ...

    @overload
    def __contains__(self, arg: object, /) -> bool: ...

    def count(self, arg: NeighborSearch.Mark, /) -> int:
        """Return number of occurrences of `arg`."""

    def remove(self, arg: NeighborSearch.Mark, /) -> None:
        """Remove first occurrence of `arg`."""

class ContactSearch:
    def __init__(self, arg: float, /) -> None: ...

    class Ignore(enum.Enum):
        Nothing = 0

        SameResidue = 1

        AdjacentResidues = 2

        SameChain = 3

        SameAsu = 4

    class Result:
        @property
        def partner1(self) -> CRA: ...

        @property
        def partner2(self) -> CRA: ...

        @property
        def image_idx(self) -> int: ...

        @property
        def dist(self) -> float: ...

    @property
    def search_radius(self) -> float: ...

    @search_radius.setter
    def search_radius(self, arg: float, /) -> None: ...

    @property
    def ignore(self) -> ContactSearch.Ignore: ...

    @ignore.setter
    def ignore(self, arg: ContactSearch.Ignore, /) -> None: ...

    @property
    def twice(self) -> bool: ...

    @twice.setter
    def twice(self, arg: bool, /) -> None: ...

    @property
    def special_pos_cutoff_sq(self) -> float: ...

    @special_pos_cutoff_sq.setter
    def special_pos_cutoff_sq(self, arg: float, /) -> None: ...

    @property
    def min_occupancy(self) -> float: ...

    @min_occupancy.setter
    def min_occupancy(self, arg: float, /) -> None: ...

    def setup_atomic_radii(self, arg0: float, arg1: float, /) -> None: ...

    def get_radius(self, arg: Element, /) -> float: ...

    def set_radius(self, arg0: Element, arg1: float, /) -> None: ...

    def find_contacts(self, arg: NeighborSearch, /) -> list[ContactSearch.Result]: ...

class LinkHunt:
    def __init__(self) -> None: ...

    class Match:
        @property
        def chem_link(self) -> ChemLink: ...

        @property
        def chem_link_count(self) -> int: ...

        @property
        def cra1(self) -> CRA: ...

        @property
        def cra2(self) -> CRA: ...

        @property
        def same_image(self) -> bool: ...

        @property
        def bond_length(self) -> float: ...

        @property
        def conn(self) -> Connection: ...

    def index_chem_links(self, monlib: MonLib, use_alias: bool = True) -> None: ...

    def find_possible_links(self, st: Structure, bond_margin: float, radius_margin: float, ignore: ContactSearch.Ignore = ContactSearch.Ignore.SameResidue) -> list[LinkHunt.Match]: ...

class BondIndex:
    def __init__(self, arg: Model, /) -> None: ...

    def add_link(self, arg0: Atom, arg1: Atom, arg2: bool, /) -> None: ...

    def add_monomer_bonds(self, arg: MonLib, /) -> None: ...

    def are_linked(self, arg0: Atom, arg1: Atom, arg2: bool, /) -> bool: ...

    def graph_distance(self, a: Atom, b: Atom, same_index: bool, max_distance: int = 4) -> int: ...

class ChemCompModel(enum.IntFlag):
    # __str__ and __repr__ disabled in stubs.pat

    # __str__ and __repr__ disabled in stubs.pat

    Xyz = 1

    Example = 2

    Ideal = 4

def read_structure(path: str, merge_chain_parts: bool = True, format: CoorFormat = CoorFormat.Unknown, save_doc: cif.Document | None = None) -> Structure:
    """Reads a coordinate file into Structure."""

def make_structure_from_block(block: cif.Block) -> Structure:
    """Takes mmCIF block and returns Structure."""

def make_structure_from_chemcomp_block(block: cif.Block, which: int = 7) -> Structure:
    """CIF block from CCD or monomer library -> single-residue Model(s)."""

@overload
def read_pdb_string(s: str, max_line_length: int = 0, ignore_ter: bool = False, split_chain_on_ter: bool = False) -> Structure:
    """Reads a string as PDB file."""

@overload
def read_pdb_string(s: bytes, max_line_length: int = 0, ignore_ter: bool = False, split_chain_on_ter: bool = False) -> Structure: ...

def read_pdb(filename: str, max_line_length: int = 0, ignore_ter: bool = False, split_chain_on_ter: bool = False) -> Structure: ...

def read_small_structure(path: str) -> SmallStructure:
    """Reads a small molecule CIF file."""

def make_small_structure_from_block(block: cif.Block) -> SmallStructure:
    """Takes CIF block and returns SmallStructure."""

def estimate_uncompressed_size(path: str) -> int:
    """Returns uncompressed size of a .gz file (not always reliable)"""

def adp_symmetry_constraints(arg: SpaceGroup, /) -> list[list[float]]: ...

class Scaling:
    def __init__(self, arg0: UnitCell, arg1: SpaceGroup, /) -> None: ...

    @property
    def cell(self) -> UnitCell: ...

    @cell.setter
    def cell(self, arg: UnitCell, /) -> None: ...

    @property
    def k_overall(self) -> float: ...

    @k_overall.setter
    def k_overall(self, arg: float, /) -> None: ...

    @property
    def b_overall(self) -> SMat33d: ...

    @b_overall.setter
    def b_overall(self, arg: SMat33d, /) -> None: ...

    @property
    def use_solvent(self) -> bool: ...

    @use_solvent.setter
    def use_solvent(self, arg: bool, /) -> None: ...

    @property
    def k_sol(self) -> float: ...

    @k_sol.setter
    def k_sol(self, arg: float, /) -> None: ...

    @property
    def b_sol(self) -> float: ...

    @b_sol.setter
    def b_sol(self, arg: float, /) -> None: ...

    @property
    def parameters(self) -> list[float]: ...

    @parameters.setter
    def parameters(self, arg: Sequence[float], /) -> None: ...

    def prepare_points(self, calc: ComplexAsuData, obs: ValueSigmaAsuData, mask: ComplexAsuData | None = None) -> None: ...

    def fit_isotropic_b_approximately(self) -> None: ...

    def fit_b_star_approximately(self) -> None: ...

    def fit_parameters(self) -> float: ...

    def calculate_r_factor(self) -> float: ...

    @overload
    def get_overall_scale_factor(self, hkl: Sequence[int]) -> float: ...

    @overload
    def get_overall_scale_factor(self, arg: Annotated[NDArray[numpy.int32], dict(shape=(None, 3), device='cpu')], /) -> NDArray[numpy.float64]: ...

    def get_solvent_scale(self, stol2: float) -> float: ...

    def scale_data(self, asu_data: ComplexAsuData, mask_data: ComplexAsuData | None = None) -> None: ...

    def scale_value(self, hkl: Sequence[int], f_value: complex, mask_value: complex) -> complex: ...

def set_leak_warnings(arg: bool, /) -> None: ...
